/* mbed I2CTextLCD Library
 * Copyright (c) 2007-2009 sford
 * Copyright (c) 2010 Wim De Roeve  changed to work with I2C PCF8575
 * Released under the MIT License: http://mbed.org/license/mit
 */

#include "I2CTextLCD.h"
#include "mbed.h"
#include "error.h"

using namespace mbed;

/*
     * useful info found at http://www.a-netz.de/lcd.en.php
     *
     *
     * Initialisation
     * ==============
     *
     * After attaching the supply voltage/after a reset, the display needs to be brought in to a defined state
     *
     * - wait approximately 15 ms so the display is ready to execute commands
     * - Execute the command 0x30 ("Display Settings") three times (wait 1,64ms after each command, the busy flag cannot be queried now).
     * - The display is in 8 bit mode, so if you have only connected 4 data pins you should only transmit the higher nibble of each command.
     * - If you want to use the 4 bit mode, now you can execute the command to switch over to this mode now.
         * - Execute the "clear display" command
     *
     * Timing
     * ======
     *
     * Nearly all commands transmitted to the display need 40us for execution.
     * Exceptions are the commands "Clear Display and Reset" and "Set Cursor to Start Position"
     * These commands need 1.64ms for execution. These timings are valid for all displays working with an
     * internal clock of 250kHz. But I do not know any displays that use other frequencies. Any time you
     * can use the busy flag to test if the display is ready to accept the next command.
     *
     */

I2CTextLCD::I2CTextLCD(PinName sda, PinName scl, int i2cAddress , int columns, int rows,
                       bool backlight) : _i2c(sda, scl) {

    _i2cAddress = i2cAddress;
    _columns = columns;
    _rows = rows;
    _backlight=backlight;
  

    // Should theoretically wait 15ms, but most things will be powered up pre-reset
    // so i'll disable that for the minute. If implemented, could wait 15ms post reset
    // instead
    // wait(0.015);

    // send "Display Settings" 3 times (Only top nibble of 0x30 as we've got 4-bit bus)

    for (int i=0; i<3; i++) {
        writeNibble(EIGHT_BITMODE,false);
        wait(0.00164);      // this command takes 1.64ms, so wait for it
    }
    writeNibble(FOUR_BITMODE,false); // 4-bit mode

    writeCommand(0x28);    // Function set 001 BW N F - -
    writeCommand(0x0C);
    writeCommand(0x6);  //  Cursor Direction and Display Shift : 0000 01 CD S (CD 0-left, 1-right S(hift) 0-no, 1-yes

    cls();


}

int I2CTextLCD::_putc(int value) {
    if (value == '\n') {
        newline();
    } else {
        writeData(value);
    }
    return value;
}

int I2CTextLCD::_getc() {
    return 0;
}

void I2CTextLCD::backlight(bool status) {
    _backlight=status;
    if (_backlight)
        writeI2CByte(BACKLIGHT_ON | E1_ON);
    else
        writeI2CByte(E1_ON);
}


void I2CTextLCD::newline() {
    _column = 0;
    _row++;
    if (_row >= _rows) {
        _row = 0;
    }
    locate(_column, _row);
}

void I2CTextLCD::locate(int column, int row) {
    if (column < 0 || column >= _columns || row < 0 || row >= _rows) {
        error("locate(%d,%d) out of range on %dx%d display", column, row, _columns, _rows);
        return;
    }

    _row = row;
    _column = column;
    int address = 0x80 + (_row * 40) + _column; // memory starts at 0x80, and is 40 chars long per row
    // pc_LCD.traceOut("locate %dx%d\r\n", column, row);
    writeCommand(address);
}

void I2CTextLCD::cls() {
    writeCommand(0x01); // Clear Display
    wait(0.00164f);     // This command takes 1.64 ms
    locate(0, 0);
}

void I2CTextLCD::reset() {
    cls();
}

void I2CTextLCD::writeNibble(int data, bool rs) {

    data = (data & 0xF);


    if (_backlight)
        data= data  | BACKLIGHT_ON;

    if (rs) {
        data = data | RS_ON; // set rs bit
    }

    writeI2CByte(data | E1_ON); // E=1

    wait(0.000040f);
    writeI2CByte(data);   // E=0

    wait(0.000040f);
    writeI2CByte(data | E1_ON);  // E=1
}

void I2CTextLCD::writeByte(int data, bool rs) {
    writeNibble(data >> 4 , rs);
    writeNibble(data >> 0 , rs);
}

void I2CTextLCD::writeCommand(int command) {
    // RS = 0;
    writeByte(command,false);
}

void I2CTextLCD::writeData(int data) {
    //RS = 1
    writeByte(data,true);

    _column++;
    if (_column >= _columns) {
        newline();
    }
}

void I2CTextLCD::writeI2CByte(int data) {
    char cmd[2];
    cmd[0] = (data & 0xFF);
    cmd[1] = (data >> 8);
    _i2c.write(_i2cAddress, cmd, 2);
}





