/**
@file BMP180.h

@brief Header file containing member functions and variables

*/

#ifndef BMP180_H
#define BMP180_H

// BMP180 address (5.2 datasheet)
#define BMP180_W_ADDRESS   0xEE
#define BMP180_R_ADDRESS   0xEF
// Register Descriptions (4 datasheet)
#define ID_REG          0xD0
#define EEPROM_REG_ADD  0xAA

// uncomment line below to run in debug mode and compare to datasheet values
//#define DEBUG

// Struct to store calibration data
typedef struct Calibration Calibration;
struct Calibration {
    int16_t AC1;
    int16_t AC2;
    int16_t AC3;
    uint16_t AC4;
    uint16_t AC5;
    uint16_t AC6;
    int16_t B1;
    int16_t B2;
    int16_t MB;
    int16_t MC;
    int16_t MD;
};

// Struct for measurement (temperature and pressure)
typedef struct Measurement Measurement;
struct Measurement {
    float temperature;
    float pressure;
};

#include "mbed.h"

/**
@brief Library for interfacing with BMP180 Barometer
@see https://www.bosch-sensortec.com/en/homepage/products_3/environmental_sensors_1/bmp180_1/bmp180
@see https://www.sparkfun.com/products/11824

@brief Revision 1.0

@author Craig A. Evans
@date   March 2015
 *
 * Example:
 * @code

#include "mbed.h"
#include "BMP180.h"

BMP180 bmp180(p28,p27);   // SDA, SCL
Serial serial(USBTX,USBRX);

int main() {

    // initiliase barometer
    bmp180.init();

    Measurement measurement;  // measurement structure declared in BMP180 class

    while(1) {

        // read values (T in Celsius and P in mb) and print over serial port
        measurement = bmp180.readValues();
        serial.printf("T = %.2f C P = %.2f mb\n",measurement.temperature,measurement.pressure);
        wait(1.0);  // short delau until next reading

    }
}


 * @endcode
 */

class BMP180
{
public:

    /** Create a BMP180 object connected to the specified I2C pins
    *
    * @param sdaPin - mbed SDA pin
    * @param sclPin - mbed SCL pin
    *
    */
    BMP180(PinName sdaPin, PinName sclPin);

    /** Initialise barometer - reads factory calibration data
    *
    */
    void init();

    /** Read current temperature and pressure values
    *
    * @returns Measurement structure. Memebers are temperature in C (float) and pressure in mbar (float)
    *
    */
    Measurement readValues();

private:
    void error();
    int32_t readUncompensatedTemperatureValue();
    int32_t readUncompensatedPressureValue();
    int32_t calcTrueTemperature(int32_t UT);
    int32_t calcTruePressure(int32_t UP);
    void sendByteToRegister(char byte,char reg);
    char readByteFromRegister(char reg);
    void readBytesFromRegister(char reg,int numberOfBytes,char bytes[]);
    void readCalibrationData();


private:  // private variables
    I2C* i2c;
    BusOut* leds;
    Calibration calibration;  // variable to store calibration data
    // variables for calculation
    int32_t X1,X2,X3,B3,B5,B6;
    uint32_t B4,B7;
    int oss;  // oversampling setting
};

#endif