/******************************************************************************
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *****************************************************************************/

/*!
 ******************************************************************************
 * @file:
 * @brief:  ADMW OS-dependent wrapper layer for GPIO interface
 *-----------------------------------------------------------------------------
 */

#include <mbed.h>

#include "inc/admw_gpio.h"
#include "inc/admw_log.h"

class GpioContext
{
public:
    GpioContext(
        PinName resetPin,
        PinName alertErrorPin,
        PinName datareadyPin)
        : _reset(resetPin),
          _alertError(alertErrorPin),
          _dataready(datareadyPin),
          _alertErrorIrq(alertErrorPin),
          _datareadyIrq(datareadyPin) {}

    ADMW_RESULT get(
        ADMW_GPIO_PIN ePinId,
        bool *pState);

    ADMW_RESULT set(
        ADMW_GPIO_PIN ePinId,
        bool state);

    ADMW_RESULT enableIrq(
        ADMW_GPIO_PIN ePinId,
        ADMW_GPIO_CALLBACK callbackFn,
        void *pArg);

    ADMW_RESULT disableIrq(
        ADMW_GPIO_PIN ePinId);

private:
    DigitalOut  _reset;

    DigitalIn   _alertError;
    DigitalIn   _dataready;

    InterruptIn _alertErrorIrq;
    InterruptIn _datareadyIrq;

    ADMW_GPIO_CALLBACK _alertErrorIrqCallback;
    ADMW_GPIO_CALLBACK _datareadyIrqCallback;

    void *_alertErrorIrqArg;
    void *_datareadyIrqArg;

    void _alertErrorIrqHandler()
    {
        _alertErrorIrqCallback(ADMW_GPIO_PIN_ALERT_ERROR, _alertErrorIrqArg);
    }
    void _datareadyIrqHandler()
    {
        _datareadyIrqCallback(ADMW_GPIO_PIN_DATAREADY, _datareadyIrqArg);
    }
};

ADMW_RESULT GpioContext::get(
    ADMW_GPIO_PIN ePinId,
    bool *pState)
{
    switch(ePinId) {
        case ADMW_GPIO_PIN_ALERT_ERROR:
            *pState = _alertError;
            return ADMW_SUCCESS;
        case ADMW_GPIO_PIN_DATAREADY:
            *pState = _dataready;
            return ADMW_SUCCESS;
        case ADMW_GPIO_PIN_RESET:
            *pState = _reset;
            return ADMW_SUCCESS;
        default:
            return ADMW_INVALID_DEVICE_NUM;
    }
}

ADMW_RESULT GpioContext::set(
    ADMW_GPIO_PIN ePinId,
    bool state)
{
    switch(ePinId) {
        case ADMW_GPIO_PIN_RESET:
            _reset = state;
            break;
        default:
            return ADMW_INVALID_DEVICE_NUM;
    }

    return ADMW_SUCCESS;
}

ADMW_RESULT GpioContext::enableIrq(
    ADMW_GPIO_PIN ePinId,
    ADMW_GPIO_CALLBACK callbackFn,
    void *pArg)
{
    switch(ePinId) {
        case ADMW_GPIO_PIN_ALERT_ERROR:
            _alertErrorIrqCallback = callbackFn;
            _alertErrorIrqArg = pArg;
            _alertErrorIrq.rise(callback(this, &GpioContext::_alertErrorIrqHandler));
            return ADMW_SUCCESS;
        case ADMW_GPIO_PIN_DATAREADY:
            _datareadyIrqCallback = callbackFn;
            _datareadyIrqArg = pArg;
            _datareadyIrq.rise(callback(this, &GpioContext::_datareadyIrqHandler));
            return ADMW_SUCCESS;
        default:
            return ADMW_INVALID_DEVICE_NUM;
    }
}

ADMW_RESULT GpioContext::disableIrq(
    ADMW_GPIO_PIN ePinId)
{
    switch(ePinId) {
        case ADMW_GPIO_PIN_ALERT_ERROR:
            _alertErrorIrq.rise(NULL);
            return ADMW_SUCCESS;
        case ADMW_GPIO_PIN_DATAREADY:
            _datareadyIrq.rise(NULL);
            return ADMW_SUCCESS;
        default:
            return ADMW_INVALID_DEVICE_NUM;
    }
}

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Open the GPIO interface and allocate resources
 */
ADMW_RESULT admw_GpioOpen(
    ADMW_PLATFORM_GPIO_CONFIG *pConfig,
    ADMW_GPIO_HANDLE *phDevice)
{
    GpioContext *pCtx = new GpioContext((PinName)pConfig->resetPin,
                                        (PinName)pConfig->alertErrorPin,
                                        (PinName)pConfig->datareadyPin);
    if (!pCtx) {
        ADMW_LOG_ERROR("Failed to allocate memory for GPIO context");
        return ADMW_NO_MEM;
    }

    *phDevice = reinterpret_cast<ADMW_GPIO_HANDLE>(pCtx);
    return ADMW_SUCCESS;
}

/*
 * Get the state of the specified GPIO pin
 */
ADMW_RESULT admw_GpioGet(
    ADMW_GPIO_HANDLE hDevice,
    ADMW_GPIO_PIN ePinId,
    bool *pbState)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    return pCtx->get(ePinId, pbState);
}

/*
 * Set the state of the specified GPIO pin
 */
ADMW_RESULT admw_GpioSet(
    ADMW_GPIO_HANDLE hDevice,
    ADMW_GPIO_PIN ePinId,
    bool bState)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    return pCtx->set(ePinId, bState);
}

/*
 * Enable interrupt notifications on the specified GPIO pin
 */
ADMW_RESULT admw_GpioIrqEnable(
    ADMW_GPIO_HANDLE hDevice,
    ADMW_GPIO_PIN ePinId,
    ADMW_GPIO_CALLBACK callback,
    void *arg)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    return pCtx->enableIrq(ePinId, callback, arg);
}

/*
 * Disable interrupt notifications on the specified GPIO pin
 */
ADMW_RESULT admw_GpioIrqDisable(
    ADMW_GPIO_HANDLE hDevice,
    ADMW_GPIO_PIN ePinId)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    return pCtx->disableIrq(ePinId);
}

/*
 * Close the GPIO interface and free resources
 */
void admw_GpioClose(
    ADMW_GPIO_HANDLE hDevice)
{
    GpioContext *pCtx = reinterpret_cast<GpioContext *>(hDevice);

    delete pCtx;
}

#ifdef __cplusplus
}
#endif
