/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 *****************************************************************************/
/*!
 ******************************************************************************
 * @file:
 *-----------------------------------------------------------------------------
 *
 */
#include "mbed.h"
#include "admw_api.h"
#include "admw1001/admw1001_api.h"
#include "admw_log.h"
#include "common/utils.h"
#include "common/platform.h"
#include "admw1001/ADMW1001_REGISTERS.h"

extern ADMW_CONFIG thermocouple_typeK_cjc0_config;
extern ADMW_CONFIG thermocouple_typeT_cjc0_config;
extern ADMW_CONFIG thermocouple_typeJ_cjc0_config;
extern ADMW_CONFIG rtd_3w_pt100_config;
extern ADMW_CONFIG rtd_4w_config;
extern ADMW_CONFIG bridge_4w_load_cell_config;
extern ADMW_CONFIG multichannel_continuous_config;
extern ADMW_CONFIG multichannel_singlecycle_config;
extern ADMW_CONFIG i2c0_sensirionSHT3X_config;
/* Change the following pointer to select any of the configurations above */
static ADMW_CONFIG *pSelectedConfig = &rtd_4w_config;

static ADMW_CONNECTION connectionInfo = PLATFORM_CONNECTION_INFO;


int main()
{
    ADMW_RESULT res;
    ADMW_STATUS status;
    ADMW_DEVICE_HANDLE hDevice;
    ADMW_MEASUREMENT_MODE eMeasurementMode = ADMW_MEASUREMENT_MODE_NORMAL;
    bool bDeviceReady;

    /*
     * Open an ADMW1001 device instance.
     */
    res = admw_Open(0, &connectionInfo, &hDevice);
    if (res != ADMW_SUCCESS) {
        ADMW_LOG_ERROR("Failed to open device instance");
        return res;
    }

    /*
     * Reset the given ADMW1001 device....
     */
    ADMW_LOG_INFO("Resetting ADMW1001 device, please wait...");
    res = admw_Reset(hDevice);
    if (res != ADMW_SUCCESS) {
        ADMW_LOG_ERROR("Failed to reset device");
        return res;
    }
    /*
     * ...and wait until the device is ready.
     */
    do {
        wait_ms(100);
        res = admw_GetDeviceReadyState(hDevice, &bDeviceReady);
        if (res != ADMW_SUCCESS) {
            ADMW_LOG_ERROR("Failed to get device ready-state");
            return res;
        }
    } while (! bDeviceReady);
    ADMW_LOG_INFO("ADMW1001 device ready");
    /*
     * Write configuration settings to the device registers.
     * Settings are not applied until admw_ApplyConfigUpdates() is called.
     */
    ADMW_LOG_INFO("Setting device configuration");
    res = admw_SetConfig(hDevice, pSelectedConfig);
    if (res != ADMW_SUCCESS) {
        ADMW_LOG_ERROR("Failed to set device configuration");
        return res;
    }
    res = admw_ApplyConfigUpdates(hDevice);
    if (res != ADMW_SUCCESS) {
        ADMW_LOG_ERROR("Failed to apply device configuration");
        return res;
    }
    /*
     * Check device status after updating the configuration
     */
    res = admw_GetStatus(hDevice, &status);
    deviceInformation(hDevice);
    if (res != ADMW_SUCCESS) {
        ADMW_LOG_ERROR("Failed to retrieve device status");
        return res;
    }
    if (status.deviceStatus &
            (ADMW_DEVICE_STATUS_ERROR | ADMW_DEVICE_STATUS_ALERT)) {
        utils_printStatus(&status);
    }

    /*
     * Kick off the measurement cycle here
     */
    ADMW_LOG_INFO("Configuration completed, starting measurement cycles");
    utils_runMeasurement(hDevice, eMeasurementMode);

    /*
     * Clean up and exit
     */
    res = admw_Close(hDevice);
    if (res != ADMW_SUCCESS) {
        ADMW_LOG_ERROR("Failed to close device instance");
        return res;
    }

    return 0;
}

