/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*!
 ******************************************************************************
 * @file:   admw1001_config.h
 * @brief:  Configuration type definitions for ADMW1001.
 *-----------------------------------------------------------------------------
 */

#ifndef __ADMW1001_CONFIG_H__
#define __ADMW1001_CONFIG_H__

#include "admw_platform.h"
#include "admw1001_sensor_types.h"

/*! @addtogroup ADMW1001_Api
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! Maximum length allowed for a digital sensor command */
#define ADMW1001_SENSOR_COMMAND_MAX_LENGTH 7

/*! ADMW1001 channel priority options */
typedef enum
{
    ADMW1001_CHANNEL_PRIORITY_0 = 0,
    ADMW1001_CHANNEL_PRIORITY_1,
    ADMW1001_CHANNEL_PRIORITY_2,
    ADMW1001_CHANNEL_PRIORITY_3,
    ADMW1001_CHANNEL_PRIORITY_4,
    ADMW1001_CHANNEL_PRIORITY_5,
    ADMW1001_CHANNEL_PRIORITY_6,
    ADMW1001_CHANNEL_PRIORITY_7,
    ADMW1001_CHANNEL_PRIORITY_8,
    ADMW1001_CHANNEL_PRIORITY_9,
    ADMW1001_CHANNEL_PRIORITY_10,
    ADMW1001_CHANNEL_PRIORITY_11,
    ADMW1001_CHANNEL_PRIORITY_12,

    ADMW1001_CHANNEL_PRIORITY_HIGHEST = ADMW1001_CHANNEL_PRIORITY_0,
    ADMW1001_CHANNEL_PRIORITY_LOWEST  = ADMW1001_CHANNEL_PRIORITY_12,

}   ADMW1001_CHANNEL_PRIORITY;

/*! ADMW1001 operating mode options */
typedef enum
{
    ADMW1001_OPERATING_MODE_SINGLECYCLE = 0,
    /*!< Executes a single measurement cycle and stops */
    ADMW1001_OPERATING_MODE_CONTINUOUS,
    /*!< Continuously executes measurement cycles */
    ADMW1001_OPERATING_MODE_MULTICYCLE,
    /*!< Executes a burst of measurement cycles, repeated at defined intervals */

}   ADMW1001_OPERATING_MODE;

/*! ADMW1001 data ready mode options */
typedef enum
{
    ADMW1001_DATAREADY_PER_CONVERSION = 0,
    /*!< The DATAREADY signal is asserted after completion of each conversion
     *   - a single data sample only from the latest completed conversion is
     *     stored in this mode
     */
    ADMW1001_DATAREADY_PER_CYCLE,
    /*!< The DATAREADY signal is asserted after completion of each measurement
     *   cycle
     *   - data samples only from the lastest completed measurement cycle are
     *     stored in this mode
     */
    ADMW1001_DATAREADY_PER_FIFO_FILL,
    /*!< The DATAREADY signal is asserted after each fill of the data FIFO
     *   - applicable only when @ref ADMW1001_OPERATING_MODE_CONTINUOUS or
     *     @ref ADMW1001_OPERATING_MODE_MULTICYCLE is also selected
     */

}   ADMW1001_DATAREADY_MODE;

/*! ADMW1001 power mode options */
typedef enum
{
    ADMW1001_POWER_MODE_ACTIVE = 0,
    /*!< Part is fully powered up and either cycling through a sequence or awaiting a configuration */
    ADMW1001_POWER_MODE_HIBERNATION,
    /*!< module has entede hibernation mode. All analog circuitry is disabled. All peripherals disabled apart from the Wake-up pin functionality. */

}   ADMW1001_POWER_MODE;

/*! ADMW1001 measurement analog filter settling options */
typedef enum
{
    ADMW1001_FILTER_SETTLING_ALWAYS = 0,
    /*!< Allow full settling time to elapse between every measurement from an analog sensor */
    ADMW1001_FILTER_SETTLING_FAST,
    /*!< Skip settling time between consecutive measurements from an analog sensor */

}   ADMW1001_FILTER_SETTLING;

/*! ADMW1001 measurement unit options
 *
 *  Optionally select a measurement unit for final conversion results.
 *  Currently applicable only to specific temperature sensor types.
 */
typedef enum
{
    ADMW1001_MEASUREMENT_UNIT_UNSPECIFIED = 0,
    /*!< No measurement unit specified */
    ADMW1001_MEASUREMENT_UNIT_CELSIUS,
    /*!< Celsius temperature unit - applicable to temperature sensors only */
    ADMW1001_MEASUREMENT_UNIT_FAHRENHEIT,
    /*!< Fahrenheit temperature unit - applicable to temperature sensors only */

}   ADMW1001_MEASUREMENT_UNIT;

typedef enum
{
    ADMW1001_BUFFER_BYPASSS_DISABLED = 0,
    /*!< Buffer Bypass Disabled */
    ADMW1001_BUFFER_BYPASSS_ENABLED,
    /*!< Buffer Bypass Enabled */

}   ADMW1001_BUFFER_BYPASSS;

/*! ADMW1001 analog input signal amplification gain options
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef enum
{
    ADMW1001_ADC_RTD_CURVE_EUROPEAN = 0,
    /*!< EUROPEAN RTD curve used. */
    ADMW1001_ADC_RTD_CURVE_AMERICAN,
    /*!< AMERICAN RTD curve used. */
    ADMW1001_ADC_RTD_CURVE_JAPANESE,
    /*!< JAPANESE RTD curve used. */
    ADMW1001_ADC_RTD_CURVE_ITS90,
    /*!< ITS90 RTD curve used. */

}   ADMW1001_ADC_RTD_CURVE;
/*! ADMW1001 analog input signal amplification gain options
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef enum
{
    ADMW1001_ADC_GAIN_1X = 0,
    /*!< no amplification gain */
    ADMW1001_ADC_GAIN_2X,
    /*!< x2 amplification gain */
    ADMW1001_ADC_GAIN_4X,
    /*!< x4 amplification gain */
    ADMW1001_ADC_GAIN_8X,
    /*!< x8 amplification gain */
    ADMW1001_ADC_GAIN_16X,
    /*!< x16 amplification gain */
    ADMW1001_ADC_GAIN_32X,
    /*!< x32 amplification gain */
    ADMW1001_ADC_GAIN_64X,
    /*!< x64 amplification gain */
    ADMW1001_ADC_GAIN_128X,
    /*!< x128 amplification gain */

}   ADMW1001_ADC_GAIN;

/*! ADMW1001 analog sensor excitation state options
 *
 *  @note applicable only to ADC analog sensor channels, and
 *        specific sensor types
 */
typedef enum
{
    ADMW1001_ADC_EXC_STATE_CYCLE_POWER = 0,
    /*!< Excitation for measurement is active only during measurement */
    ADMW1001_ADC_EXC_STATE_ALWAYS_ON,
    /*!< Excitation for measurement is always on */

}   ADMW1001_ADC_EXC_STATE;

/*! ADMW1001 analog sensor excitation current output level options
 *
 *  @note applicable only to ADC analog sensor channels, and
 *        specific sensor types
 */
typedef enum
{
    ADMW1001_ADC_NO_EXTERNAL_EXC_CURRENT = -1,
    /*!< NO External excitation is provided */
    ADMW1001_ADC_EXC_CURRENT_EXTERNAL = 0,
    /*!< External excitation is provided */
    ADMW1001_ADC_EXC_CURRENT_50uA,
    /*!< 50uA excitation current enabled */
    ADMW1001_ADC_EXC_CURRENT_100uA,
    /*!< 100uA excitation current  */
    ADMW1001_ADC_EXC_CURRENT_250uA,
    /*!< 250uA excitation current enabled */
    ADMW1001_ADC_EXC_CURRENT_500uA,
    /*!< 500uA excitation current enabled */
    ADMW1001_ADC_EXC_CURRENT_1000uA,
    /*!< 1mA excitation current enabled */

}   ADMW1001_ADC_EXC_CURRENT;

/*! ADMW1001 analog sensor excitation current ratios used for diode sensor
 *
 *  @note applicable only to a diode sensor
 */
typedef enum
{
    ADMW1001_ADC_EXC_CURRENT_IOUT_DIODE_10UA_100UA = 0,
    /**< 2 Current measurement 10uA 100uA        */
    ADMW1001_ADC_EXC_CURRENT_IOUT_DIODE_20UA_160UA,
    /**< 2 Current measurement 20uA 160uA        */
    ADMW1001_ADC_EXC_CURRENT_IOUT_DIODE_50UA_300UA,
    /**< 2 Current measurement 50uA 300uA        */
    ADMW1001_ADC_EXC_CURRENT_IOUT_DIODE_100UA_600UA,
    /**< 2 Current measurement 100uA 600uA       */
    ADMW1001_ADC_EXC_CURRENT_IOUT_DIODE_10UA_50UA_100UA,
    /**< 3 current measuremetn 10uA 50uA 100uA   */
    ADMW1001_ADC_EXC_CURRENT_IOUT_DIODE_20UA_100UA_160UA,
    /**< 3 current measuremetn 20uA 100uA 160uA  */
    ADMW1001_ADC_EXC_CURRENT_IOUT_DIODE_50UA_150UA_300UA,
    /**< 3 current measuremetn 50uA 150uA 300uA  */
    ADMW1001_ADC_EXC_CURRENT_IOUT_DIODE_100UA_300UA_600UA,
    /**< 3 current measuremetn 100uA 300uA 600uA */

}   ADMW1001_ADC_EXC_CURRENT_DIODE_RATIO;

/*! ADMW1001 analog reference selection options
 *
 *  @note applicable only to ADC analog sensor channels, and
 *        specific sensor types
 */
typedef enum
{
    ADMW1001_ADC_REFERENCE_VOLTAGE_INTERNAL = 0,
    /*!< Internal VRef - 1.2V */
    ADMW1001_ADC_REFERENCE_VOLTAGE_EXTERNAL_1 = 1,
    /*!< External reference voltage #1 */
    ADMW1001_ADC_REFERENCE_VOLTAGE_AVDD = 3,
    /*!< Analog Supply Voltage AVDD reference (typically 3.3V) is selected */

}   ADMW1001_ADC_REFERENCE_TYPE;

/*! ADMW1001 ADC Reference configuration
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef enum
{
    ADMW1001_ADC_GND_SW_OPEN = 0,
    /*!< Ground switch not enabled for measurement. */
    ADMW1001_ADC_GND_SW_CLOSED,
    /*!< Ground switch enabled for measurement. */

}   ADMW1001_ADC_GND_SW;

/*! ADMW1001 analog filter chop mode
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef enum
{
    ADMW1001_CHOP_MD_NONE = 0,
    /*!< No chop performed. */
    ADMW1001_CHOP_MD_HW,
    /*!< Hardware only chop performed. */
    ADMW1001_CHOP_MD_SW,
    /*!< Software only chop performed. */
    ADMW1001_CHOP_MD_HWSW,
    /*!< Hardware and software chop performed. */

}   ADMW1001_CHOP_MD;

/*! ADMW1001 analog filter selection options
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef enum
{
    ADMW1001_ADC_FILTER_SINC4 = 0,
    /*!< SINC4 - 4th order sinc response filter */
    ADMW1001_ADC_FILTER_SINC3,
    /*!< SINC3 - 3rd order sinc response filter */

}   ADMW1001_ADC_FILTER_TYPE;

/*! ADMW1001 Sinc Filter range (SF)
 *
 *  @note applicable only to ADC analog sensor channels
 *  @note SF must be set in conjunction with chop mode
 *  and sinc filter type to achieve the desired sampling rate.
 */
typedef enum
{
    ADMW1001_SF_976HZ  = 0,
    /*!< SF setting for 976Hz sample rate. */
    ADMW1001_SF_488HZ  = 1,
    /*!< SF setting for 488Hz sample rate. */
    ADMW1001_SF_244HZ  = 3,
    /*!< SF setting for 244Hz sample rate. */
    ADMW1001_SF_122HZ  = 7,
    /*!< SF setting for 122Hz sample rate. */
    ADMW1001_SF_61HZ   = 31,
    /*!< SF setting for 61Hz sample rate. */
    ADMW1001_SF_30P5HZ = 51,
    /*!< SF setting for 61Hz sample rate. */
    ADMW1001_SF_10HZ   = 124,
    /*!< SF setting for 10Hz sample rate. */
    ADMW1001_SF_8P24HZ = 125,
    /*!< SF setting for 8.24Hz sample rate. */
    ADMW1001_SF_5HZ    = 127,
    /*!< SF setting for 5Hz sample rate. */

}   ADMW1001_SINC_FILTER_RANGE;

/*! ADMW1001 I2C clock speed options
 *
 *  @note applicable only for I2C sensors
 */
typedef enum
{
    ADMW1001_DIGITAL_SENSOR_COMMS_I2C_CLOCK_SPEED_100K = 0,
    /*!< 100kHz I2C clock speed */
    ADMW1001_DIGITAL_SENSOR_COMMS_I2C_CLOCK_SPEED_400K,
    /*!< 400kHz I2C clock speed */

}   ADMW1001_DIGITAL_SENSOR_COMMS_I2C_CLOCK_SPEED;

/*! ADMW1001 SPI mode options
 *
 *  @note applicable only for SPI sensors
 */
typedef enum
{
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_MODE_0 = 0,
    /*!< SPI mode 0 Clock Polarity = 0 Clock Phase = 0 */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_MODE_1,
    /*!< SPI mode 0 Clock Polarity = 0 Clock Phase = 1 */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_MODE_2,
    /*!< SPI mode 0 Clock Polarity = 1 Clock Phase = 0 */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_MODE_3,
    /*!< SPI mode 0 Clock Polarity = 1 Clock Phase = 1 */

}   ADMW1001_DIGITAL_SENSOR_COMMS_SPI_MODE;

/*! ADMW1001 SPI clock speed options
 *
 *  @note applicable only for SPI sensors
 */
typedef enum
{
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_8MHZ = 0,
    /*!< SPI Clock Speed configured to 8MHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_4MHZ,
    /*!< SPI Clock Speed configured to 4MHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_2MHZ,
    /*!< SPI Clock Speed configured to 2MHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_1MHZ,
    /*!< SPI Clock Speed configured to 1MHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_500KHZ,
    /*!< SPI Clock Speed configured to 500kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_250KHZ,
    /*!< SPI Clock Speed configured to 250kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_125KHZ,
    /*!< SPI Clock Speed configured to 125kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_62P5KHZ,
    /*!< SPI Clock Speed configured to 62.5kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_31P3KHZ,
    /*!< SPI Clock Speed configured to 31.3kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_15P6KHZ,
    /*!< SPI Clock Speed configured to 15.6kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_7P8KHZ,
    /*!< SPI Clock Speed configured to 7.8kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_3P9KHZ,
    /*!< SPI Clock Speed configured to 3.9kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_1P9KHZ,
    /*!< SPI Clock Speed configured to 1.9kHz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_977HZ,
    /*!< SPI Clock Speed configured to 977Hz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_488HZ,
    /*!< SPI Clock Speed configured to 488Hz */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK_244HZ,
    /*!< SPI Clock Speed configured to 244Hz */

}   ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK;

/*! ADMW1001 Power Configuration options */
typedef struct
{
    ADMW1001_POWER_MODE        powerMode;
    /*!< Power mode selection */

}   ADMW1001_POWER_CONFIG;

/*! ADMW1001 Multi-Cycle Configuration options
 *
 * @note required only when ADMW1001_OPERATING_MODE_MULTICYCLE is selected
 *       as the operatingMode (@ref ADMW1001_MEASUREMENT_CONFIG)
 */
typedef struct
{
    uint32_t    cyclesPerBurst;
    /*!< Number of cycles to complete for a single burst */
    uint32_t    burstInterval;
    /*!< Interval, in seconds, between each successive burst of cycles */

}   ADMW1001_MULTICYCLE_CONFIG;

/*! ADMW1001 Measurement Configuration options */
typedef struct
{
    ADMW1001_OPERATING_MODE    operatingMode;
    /*!< Operating mode - specifies how measurement cycles are scheduled */
    ADMW1001_DATAREADY_MODE    dataReadyMode;
    /*!< Data read mode - specifies how output samples are stored for reading */
    ADMW1001_MULTICYCLE_CONFIG multiCycleConfig;
    /*!< Multi-Cycle configuration - specifies how bursts of measurement cycles
     *   are scheduled.  Applicable only when operatingMode is
     *   ADMW1001_OPERATING_MODE_MULTICYCLE
     */
    ADMW1001_FILTER_SETTLING   filterSettling;
    /*!< Analog filter settling - specifies the policy for settling time
     *   between consecutive measurements from an analog channel in a multi-
     *   channel configuration.  Note that, in single analog channel
     *   configurations, settling time is always skipped between consecutive
     *   measurements in a measurement cycle.
     */
    uint8_t                      excitationState;
    /*!< Excitation current state */
    uint8_t                         groundSwitch;
    /*!< Option to open or close sensor ground switch */
    uint8_t                     fifoNumCycles;
    /*!< Specifies the number of cycles to fill a FIFO buffer
     *   Applicable only when operatingMode is ADMW1001_OPERATING_MODE_CONTINUOUS
     */
    uint32_t                    cycleInterval;
    /*!< Cycle interval - specifies the time interval between the start of each
     *   successive measurement cycle.  Applicable only when operatingMode is
     *   not ADMW1001_OPERATING_MODE_SINGLECYCLE
     */
    bool                        vBiasEnable;
    /*!< Enable voltage Bias output of ADC
     */
    float32_t                   externalRef1Value;
    /*!< Voltage value connected to external reference input #1.
     *   Applicable only if the selected reference type is
     *   voltage.
     *   (see @ref ADMW1001_ADC_REFERENCE_TYPE)
     */
    float32_t                   RSenseValue;
    /*!< Sense resistor value in Ohms. */
    float32_t                   externalRefVoltage;
    /*!< External Reference Voltage.
     */
    float32_t                   AVDDVoltage;
    /*!< AVDD Voltage.
     */
    uint32_t                    reserved1[1];
    /*!< Reserved for future use and ensure word alignment.
     */

}   ADMW1001_MEASUREMENT_CONFIG;

/*! ADMW1001 ADC Excitation Current output configuration
 *
 *  @note applicable only to ADC analog sensor channels, and
 *        specific sensor types
 */
typedef struct
{
    ADMW1001_ADC_EXC_CURRENT              outputLevel;
    /*!< Excitation current output level */
    ADMW1001_ADC_EXC_CURRENT_DIODE_RATIO  diodeRatio;
    /*!< Excitation current output diode ratio */
    float32_t                             idealityRatio;

}   ADMW1001_ADC_EXC_CURRENT_CONFIG;

/*! ADMW1001 ADC Filter configuration
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef struct
{
    ADMW1001_ADC_FILTER_TYPE    type;
    /*!< Filter type selection */
    ADMW1001_SINC_FILTER_RANGE  sf;
    /*!< SF value used along with filter type and chop mode to determine speed */
    ADMW1001_CHOP_MD            chopMode;
    /*!< Enable filter chop */
    bool                        notch1p2;
    /*!< Enable Notch 2 Filter Mode */

}   ADMW1001_ADC_FILTER_CONFIG;

/*! ADMW1001 ADC analog channel configuration details
 *
 *  @note applicable only to ADC analog sensor channels
 */
typedef struct
{
    ADMW1001_ADC_SENSOR_TYPE        sensor;
    /*!< Sensor type selection */
    ADMW1001_ADC_RTD_CURVE          rtdCurve;
    /*!< Rtd curve selection */
    ADMW1001_ADC_GAIN               gain;
    /*!< ADC Gain selection */
    ADMW1001_ADC_EXC_CURRENT_CONFIG current;
    /*!< ADC Excitation Current configuration */
    ADMW1001_ADC_FILTER_CONFIG      filter;
    /*!< ADC Filter configuration */
    ADMW1001_ADC_REFERENCE_TYPE     reference;
    /*!< ADC Reference configuration */
    uint8_t                         bufferBypass;
     /*!< Buffer Bypass configuration */
    uint8_t                         reserved0[2];
    /*!< Reserved for future use and ensure word alignment. */
    uint32_t                        reserved1[6];
    /*!< Reserved for future use and ensure word alignment. */

}   ADMW1001_ADC_CHANNEL_CONFIG;

/*! ADMW1001 look-up table selection
 *  Select table used to linearise the measurement.
 */
typedef enum
{
    ADMW1001_LUT_DEFAULT  = 0,
    /*!< Default LUT */
    ADMW1001_LUT_CUSTOM   = 1,
    /*!< User defined custom LUT */
    ADMW1001_LUT_RESERVED = 2,
    /*!< Reserved for future use */

}   ADMW1001_LUT_SELECT;

/*! ADMW1001 digital sensor data encoding
 *
 *  @note applicable only to SPI and I2C digital sensor channels
 */
typedef enum
{
    ADMW1001_DIGITAL_SENSOR_DATA_CODING_NONE = 0,
    /**< None/Invalid - data format is ignored if coding is set to this value */
    ADMW1001_DIGITAL_SENSOR_DATA_CODING_UNIPOLAR,
    /**< Unipolar - unsigned integer values */
    ADMW1001_DIGITAL_SENSOR_DATA_CODING_TWOS_COMPLEMENT,
    /**< Twos-complement - signed integer values */
    ADMW1001_DIGITAL_SENSOR_DATA_CODING_OFFSET_BINARY,
    /**< Offset Binary - used to represent signed values with unsigned integers,
     *   with the mid-range value representing 0 */

}   ADMW1001_DIGITAL_SENSOR_DATA_CODING;

/*! ADMW1001 digital sensor data format configuration
 *
 *  @note applicable only to SPI and I2C digital sensor channels
 */
typedef struct
{
    ADMW1001_DIGITAL_SENSOR_DATA_CODING coding;
    /**< Data Encoding of Sensor Result */
    bool littleEndian;
    /**< Set as true if data format is little-endian, false otherwise */
    bool leftJustified;
    /**< Set as true if data is left-justified in the data frame, false otherwise */
    uint8_t frameLength;
    /**< Data frame length (number of bytes to read from the sensor) */
    uint8_t numDataBits;
    /**< Number of relevant data bits to extract from the data frame */
    uint8_t bitOffset;
    /**< Data bit offset, relative to data alignment within the data frame */
    uint8_t reserved[2];
    /*!< Reserved for future use and ensure word alignment. */

}   ADMW1001_DIGITAL_SENSOR_DATA_FORMAT;

/*! ADMW1001 digital sensor command
 *
 *  @note applicable only to SPI and I2C digital sensor channels
 */
typedef struct
{
    uint8_t command[ADMW1001_SENSOR_COMMAND_MAX_LENGTH];
    /*!< Optional command bytes to send to the device */
    uint8_t commandLength;
    /*!< Number of valid command bytes.  Set to 0 if unused */

}   ADMW1001_DIGITAL_SENSOR_COMMAND;

/*! ADMW1001 digital sensor calibration param
 *
 *  @note applicable only to digital sensor channels
 */
typedef struct
{
    uint32_t calibrationParam;
    /*!< Independently established environmental variable used during calibration
     *   of a digital sensor. Used only if the sensor supports calibration
     *   and expects an environmental parameter
     */
    bool enableCalibrationParam;
    /*!< Allow Calibration_Parameter to be used during calibration of any digital sensor */
    uint8_t                         reserved[3];
    /*!< Reserved for future use and ensure word alignment. */

}   ADMW1001_DIGITAL_CALIBRATION_COMMAND;

/*! ADMW1001 digital sensor communication config
 *
 *  @note applicable only to digital sensor channels
 */
typedef struct
{
    bool                                            useCustomCommsConfig;
    /*!< Optional parameter to enable user digital communication settings */
    ADMW1001_DIGITAL_SENSOR_COMMS_I2C_CLOCK_SPEED   i2cClockSpeed;
    /*!< Optional parameter to configure specific i2c speed for i2c sensor */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_MODE          spiMode;
    /*!< Optional parameter to configure specific spi mode for spi sensor */
    ADMW1001_DIGITAL_SENSOR_COMMS_SPI_CLOCK         spiClock;
    /*!< Optional parameter to configure specific spi clock for spi sensor */

}   ADMW1001_DIGITAL_SENSOR_COMMS;

/*! ADMW1001 I2C digital channel configuration details
 *
 *  @note applicable only to I2C digital sensor channels
 */
typedef struct
{
    ADMW1001_I2C_SENSOR_TYPE                sensor;
    /*!< Sensor type selection */
    uint8_t                                 deviceAddress;
    /*!< I2C device address (7-bit) */
    uint8_t                                 reserved;
    /*!< Reserved for future use and ensure word alignment. */
    ADMW1001_DIGITAL_SENSOR_COMMAND         configurationCommand;
    /*!< Optional configuration command to send to the device at start-up.
     *   A default configuration command will be used if this is not specified.
     *   Applicable only to specific I2C sensor types.
     */
    ADMW1001_DIGITAL_SENSOR_COMMAND         dataRequestCommand;
    /*!< Optional data request command to send to the device for each sample.
     *   A default data request command will be used if this is not specified.
     *   Applicable only to specific I2C sensor types.
     */
    ADMW1001_DIGITAL_SENSOR_DATA_FORMAT     dataFormat;
    /*!< Optional data format configuration to parse/extract data from the device.
     *   A default data format will be used if this is not specified.
     *   Applicable only to specific I2C sensor types
     */
    ADMW1001_DIGITAL_CALIBRATION_COMMAND    digitalCalibrationParam;
    /*!< This is used to supply an independently established environmental variable
     *   that must be used during calibration of a [digital] sensor which a)
     *   supports calibration and b) expects an environmental parameter.
     *   An example is a CO2 sensor, which may require the CO2 concentration level
     *   when performing a calibration
     */
    ADMW1001_DIGITAL_SENSOR_COMMS           configureComms;
    /*!< Optional configuration to setup a user communication config.
     *   A default configuration  will be used if this is not specified.
     *   Applicable only to specific I2C sensor types.
     */

}   ADMW1001_I2C_CHANNEL_CONFIG;

/*! ADMW1001 SPI digital channel configuration details
 *
 *  @note applicable only to SPI digital sensor channels
 */
typedef struct
{
    ADMW1001_SPI_SENSOR_TYPE        sensor;
    /*!< Sensor type selection */
    uint8_t                         reserved[2];
    /*!< Reserved for future use and ensure word alignment. */
    ADMW1001_DIGITAL_SENSOR_COMMAND configurationCommand;
    /*!< Optional configuration command to send to the device at start-up.
     *   A default configuration command will be used if this is not specified.
     *   Applicable only to specific SPI sensor types.
     */
    ADMW1001_DIGITAL_SENSOR_COMMAND dataRequestCommand;
    /*!< Optional data request command to send to the device for each sample.
     *   A default data request command will be used if this is not specified.
     *   Applicable only to specific SPI sensor types.
     */
    ADMW1001_DIGITAL_SENSOR_DATA_FORMAT dataFormat;
    /*!< Optional data format configuration to parse/extract data from the device.
     *   A default data format will be used if this is not specified.
     *   Applicable only to specific SPI sensor types
     */
    ADMW1001_DIGITAL_CALIBRATION_COMMAND digitalCalibrationParam;
    /*!< This is used to supply an independently established environmental variable
     *   that must be used during calibration of a [digital] sensor which a)
     *   supports calibration and b) expects an environmental parameter.
     *   An example is a CO2 sensor, which may require the CO2 concentration level
     *   when performing a calibration
     */
    ADMW1001_DIGITAL_SENSOR_COMMS       configureComms;
    /*!< Optional configuration to setup a user communication config.
     *   A default configuration  will be used if this is not specified.
     *   Applicable only to specific SPI sensor types.
     */

}   ADMW1001_SPI_CHANNEL_CONFIG;

/*! ADMW1001 Measurement Channel configuration details */
typedef struct
{
    bool                                enableChannel;
    /*!< Option to include this channel in normal measurement cycles */
    bool                                disablePublishing;
    /*!< Option to disable publishing of data samples from this channel.  The
     *   channel may still be included in measurement cycles, but data samples
     *   obtained from this channel will not be published.  This is typically
     *   used for channels which are required only as a compensation reference
     *   for another channel (e.g. Cold-Junction Compensation channels).
     */
    ADMW1001_CH_ID                      compensationChannel;
    /*!< Optional compensation channel.  Set to ADMW1001_CH_ID_NONE if not
     *   required.  Typically used for thermocouple sensors that require a
     *   separate measurement of the "cold-junction" temperature, which can be
     *   be provided by an RTD temperature sensor connected on a separate
     *   "compensation channel" */
    ADMW1001_LUT_SELECT                 lutSelect;
    /*!<Select Look Up Table LUT for calculations, this implies that the
     * fundamental measurement for the sensor (typically mV or Ohms)
     * 0 = default, 1= unity, 2 = custom
     */
    ADMW1001_MEASUREMENT_UNIT           measurementUnit;
    /*!< Optional measurement unit selection for conversion results.  Applicable
     *   only for certain sensor types.  Set to
     *   ADMW1001_MEASUREMENT_UNIT_DEFAULT if not applicable.
     */
    float32_t                           lowThreshold;
    /*!< Optional minimum threshold value for each processed sample, to be
     *   checked prior to publishing.  A channel ALERT condition is raised
     *   if the processed value is lower than this threshold.  Set to NaN
     *   if not required.
     */
    float32_t                           highThreshold;
    /*!< Optional maximum threshold value for each processed sample, to be
     *   checked prior to publishing.  A channel ALERT condition is raised
     *   if the processed value is higher than this threshold.  Set to NaN
     *   if not required.
     */
    float32_t                           offsetAdjustment;
    /*!< Optional offset adjustment value applied to each processed sample.
     *   Set to NaN or 0.0 if not required.
     */
    float32_t                           gainAdjustment;
    /*!< Optional gain adjustment value applied to each processed sample.
     *   Set to NaN or 1.0 if not required.
     */
    float32_t                           sensorParameter;
    /*!< Optional sensor parameter adjustment.
     *   Set to NaN or 0 if not required.
     */
    uint32_t                            measurementsPerCycle;
    /*!< The number of measurements to obtain from this channel within each
     *   cycle.  Each enabled channel is measured in turn, until the number of
     *   measurements requested for the channel has been reached.  A different
     *   number of measurements-per-cycle may be specified for each channel.
     */
    uint32_t                            cycleSkipCount;
    /*!< Optional number of cycles to skip, such that this channel is included
     *   in the sequence in only one of every (cycleSkipCount + 1) cycles that
     *   occur.  If set to 0 (default), this channel is included in every cycle;
     *   if set to 1, this channel is included in every 2nd cycle; if set to 2,
     *   this channel is included in every 3rd cycle, and so on.
     */
    uint32_t                            extraSettlingTime;
    /*!< A minimum settling time is applied internally for each channel, based
     *   on the sensor type.  However, additional settling time (milliseconds)
     *   can optionally be specified.  Set to 0 if not required.
     */
    ADMW1001_CHANNEL_PRIORITY           priority;
    /*!< By default, channels are arranged in the measurement sequence based on
     *   ascending order of channel ID.  However, a priority-level may be
     *   specified per channel to force a different ordering of the channels,
     *   with higher-priority channels appearing before lower-priority channels.
     *   Channels with equal priority are ordered by ascending order of channel
     *   ID.  Lower numbers indicate higher priority, with 0 being the highest.
     *   Set to 0 if not required.
     */
    union
    {
        ADMW1001_ADC_CHANNEL_CONFIG     adcChannelConfig;
        /*!< ADC channel configuration - applicable only to ADC channels */
        ADMW1001_I2C_CHANNEL_CONFIG     i2cChannelConfig;
        /*!< I2C channel configuration - applicable only to I2C channels */
        ADMW1001_SPI_CHANNEL_CONFIG     spiChannelConfig;
        /*!< SPI channel configuration - applicable only to SPI channels */
    };
    /*!< Only one of adcChannelConfig, i2cChannelConfig, spiChannelConfig
     *  is required, depending on the channel designation
     *   (analog, I2C, SPI)
     */

}   ADMW1001_CHANNEL_CONFIG;

/*! ADMW1001 Diagnostics configuration details */
typedef struct
{
    bool                                disableGlobalDiag;
    /*!< Option to disable the following diagnostic checks on the ADC:
     *   - Reference Detection errors
     *   - Input under-/over-voltage errors
     *   - Calibration, Conversion and Saturation errors
     */
    bool                                disableMeasurementDiag;
    /*!< Option to disable additional checks per measurement channel:
     *   - High/low threshold limit violation
     */

    bool                                disableCriticalTempAbort;
    /*!< Option to disable abort of measurement cycle if the operating
     *   temperature of the ADMW1001 has exceeded critical limits
     */

    uint8_t    osdFrequency;
    /*!< Option to enable Open-Circuit Detection at a selected cycle interval */

}   ADMW1001_DIAGNOSTICS_CONFIG;

typedef uint16_t ADMW1001_ADVANCED_ACCESS_KEY;

/*! ADMW1001 Device configuration details */
typedef struct
{
    ADMW1001_POWER_CONFIG           power;
    /*!< Power configuration details */
    ADMW1001_MEASUREMENT_CONFIG     measurement;
    /*!< Measurement configuration details */
    ADMW1001_DIAGNOSTICS_CONFIG     diagnostics;
    /*!< FFT configuration details */
    ADMW1001_CHANNEL_CONFIG         channels[ADMW1001_MAX_CHANNELS];
    /*!< Channel-specific configuration details */
    ADMW1001_ADVANCED_ACCESS_KEY    advancedAccessKey;
    /*!< Key to enable access to advanced sensor configuration options */

}   ADMW1001_CONFIG;

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADMW1001_CONFIG_H__ */
