/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:   admw_gpio.h
 * @brief:  ADMW OS-dependent wrapper layer for GPIO interface
 *-----------------------------------------------------------------------------
 */

#ifndef __ADMW_GPIO_H__
#define __ADMW_GPIO_H__

#include "inc/admw_types.h"
#include "inc/admw_platform.h"

/*! @ingroup ADMW_Host */

/*! @addtogroup ADMW_Gpio ADMW Host GPIO interface functions
 *  @{
 */

/*! GPIO pin identifiers */
typedef enum
{
    ADMW_GPIO_PIN_RESET = 0, /*!< RESET GPIO output signal */
    ADMW_GPIO_PIN_ALERT_ERROR,     /*!< ALERT GPIO input signal */
    ADMW_GPIO_PIN_DATAREADY, /*!< DATAREADY GPIO input signal */
    ADMW_GPIO_PIN_CS, /*!< CS GPIO output signal */
} ADMW_GPIO_PIN;

/*!
 * GPIO callback function signature
 *
 * @param[in] ePinId The GPIO pin which triggered the interrupt notification
 * @param[in] pArg   Optional opaque parameter to be passed to the callback
 */
typedef void (*ADMW_GPIO_CALLBACK)(
    ADMW_GPIO_PIN               ePinId,
    void                           * pArg);

/*! A handle used in all API functions to identify the GPIO interface context */
typedef void* ADMW_GPIO_HANDLE;

#ifdef __cplusplus
extern "C"
{
#endif

/*!
 * @brief Open the SPI interface and allocate resources
 *
 * @param[in]  pConfig  Pointer to platform-specific GPIO interface details
 * @param[out] phDevice Pointer to return a GPIO interface context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully
 *         - #ADMW_NO_MEM  Failed to allocate memory for interface context
 */
ADMW_RESULT admw_GpioOpen(
    ADMW_PLATFORM_GPIO_CONFIG * pConfig,
    ADMW_GPIO_HANDLE          * phDevice);

/*!
 * @brief Close GPIO interface and free resources
 *
 * @param[in] hDevice GPIO interface context handle (@ref admw_GpioOpen)
 */
void admw_GpioClose(
    ADMW_GPIO_HANDLE            hDevice);

/*!
 * @brief Get the state of the specified GPIO pin
 *
 * @param[in]  hDevice GPIO interface context handle (@ref admw_GpioOpen)
 * @param[in]  ePinId  GPIO pin to be read
 * @param[out] pbState Pointer to return the state of the GPIO pin
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully
 *         - #ADMW_INVALID_DEVICE_NUM Invalid GPIO pin specified
 */
ADMW_RESULT admw_GpioGet(
    ADMW_GPIO_HANDLE            hDevice,
    ADMW_GPIO_PIN               ePinId,
    bool                           * pbState);

/*!
 * @brief Set the state of the specified GPIO pin
 *
 * @param[in] hDevice GPIO interface context handle (@ref admw_GpioOpen)
 * @param[in] ePinId  GPIO pin to be set
 * @param[in] bState  The state to set for GPIO pin
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully
 *         - #ADMW_INVALID_DEVICE_NUM Invalid GPIO pin specified
 */
ADMW_RESULT admw_GpioSet(
    ADMW_GPIO_HANDLE            hDevice,
    ADMW_GPIO_PIN               ePinId,
    bool                             bState);

/*!
 * @brief Enable interrupt notifications on the specified GPIO pin
 *
 * @param[in] hDevice  GPIO interface context handle (@ref admw_GpioOpen)
 * @param[in] ePinId   GPIO pin on which to enable interrupt notifications
 * @param[in] callback Callback function to invoke when the GPIO is asserted
 * @param[in] arg      Optional opaque parameter to be passed to the callback
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully
 *         - #ADMW_INVALID_DEVICE_NUM Invalid GPIO pin specified
 */
ADMW_RESULT admw_GpioIrqEnable(
    ADMW_GPIO_HANDLE            hDevice,
    ADMW_GPIO_PIN               ePinId,
    ADMW_GPIO_CALLBACK          callback,
    void                           * arg);

/*!
 * @brief Disable interrupt notifications on the specified GPIO pin
 *
 * @param[in] hDevice  GPIO interface context handle (@ref admw_GpioOpen)
 * @param[in] ePinId   GPIO pin on which to disable interrupt notifications
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully
 *         - #ADMW_INVALID_DEVICE_NUM Invalid GPIO pin specified
 */
ADMW_RESULT admw_GpioIrqDisable(
    ADMW_GPIO_HANDLE            hDevice,
    ADMW_GPIO_PIN               ePinId);

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADMW_GPIO_H__ */
