/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*!
 ******************************************************************************
 * @file:   admw_1001_api.h
 * @brief:  ADMW1001 Host Library Application Programming Interface (API)
 *-----------------------------------------------------------------------------
 */

#ifndef __ADMW1001_API_H__
#define __ADMW1001_API_H__

#include "inc/admw_types.h"
#include "inc/admw_config_types.h"
#include "inc/admw_platform.h"
#include "inc/admw1001/admw1001_config.h"
#include "inc/admw1001/admw1001_lut_data.h"

#define MAXIMUM_MEASUREMENTS_PER_CYCLE 170
/*! @ingroup ADMW_Api */

/*! @defgroup ADMW1001_Api ADMW1001 Host Library API
 *  ADMW1001 device-specific API function prototypes.
 *  These are supplementary to the common ADMW Host Library API.
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*!
 * @brief Read one or more device registers at the specified register address.
 *
 * @param[in]  hDevice    ADMW1001 device context handle
 * @param[in]  nAddress   Register map address to read from
 * @param[out] pData      Pointer to return the register map data
 * @param[in]  nLength    Number of bytes of data to read from the register map
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Provides direct byte-level read access to the device register map.
 *          The size and format of the register(s) must be known.
 *
 * @note Reads from special "keyhole" or "FIFO" registers will be handled
 *       according to documentation for those registers.
 */
ADMW_RESULT admw1001_ReadRegister(
    ADMW_DEVICE_HANDLE  const hDevice,
    uint16_t            const nAddress,
    void                *const pData,
    unsigned            const nLength);
/*!
 * @brief Read one or more device registers at the specified debug register address.
 *
 * @param[in]  hDevice    ADMW1001 device context handle
 * @param[in]  nAddress   Register map address to read from
 * @param[out] pData      Pointer to return the register map data
 * @param[in]  nLength    Number of bytes of data to read from the register map
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Provides direct byte-level read access to the device register map.
 *          The size and format of the register(s) must be known.
 *
 * @note Reads from special "keyhole" or "FIFO" registers will be handled
 *       according to documentation for those registers.
 */
ADMW_RESULT admw1001_Read_Debug_Register(
    ADMW_DEVICE_HANDLE  const hDevice,
    uint16_t            const nAddress,
    void                *const pData,
    unsigned            const nLength);

/*!
 * @brief Write one or more device registers at the specified register address.
 *
 * @param[in]  hDevice    ADMW1001 device context handle
 * @param[in]  nAddress   Register map address to read from
 * @param[out] pData      Pointer to return the register map data
 * @param[in]  nLength    Number of bytes of data to read from the register map
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Provides direct byte-level write access to the device register map.
 *          The size and format of the register(s) must be known.
 *
 * @note Writes to read-only registers will be ignored by the device.
 * @note Writes to special "keyhole" registers will be handled according to
 *       documentation for those registers.
 */
ADMW_RESULT admw1001_WriteRegister(
    ADMW_DEVICE_HANDLE  const hDevice,
    uint16_t            const nAddress,
    void                *const pData,
    unsigned            const nLength);

/*!
 * @brief Write one or more device registers at the specified debug register address.
 *
 * @param[in]  hDevice    ADMW1001 device context handle
 * @param[in]  nAddress   Register map address to read from
 * @param[out] pData      Pointer to return the register map data
 * @param[in]  nLength    Number of bytes of data to read from the register map
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Provides direct byte-level write access to the device register map.
 *          The size and format of the register(s) must be known.
 *
 */
ADMW_RESULT admw1001_Write_Debug_Register(
    ADMW_DEVICE_HANDLE  const hDevice,
    uint16_t            const nAddress,
    void                *const pData,
    unsigned            const nLength);
/*
 * @brief Update power configuration settings on the device.
 *
 * @param[in] hDevice      ADMW1001 device context handle
 * @param[in] pPowerConfig Power configuration details
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetPowerConfig(
    ADMW_DEVICE_HANDLE      hDevice,
    ADMW1001_POWER_CONFIG   *pPowerConfig);

/*!
 * @brief Update measurement configuration settings on the device.
 *
 * @param[in] hDevice            ADMW1001 device context handle
 * @param[in] pMeasurementConfig Measurement configuration details
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetMeasurementConfig(
    ADMW_DEVICE_HANDLE          hDevice,
    ADMW1001_MEASUREMENT_CONFIG *pMeasurementConfig);

/*!
 * @brief Update diagnostics configuration settings on the device.
 *
 * @param[in] hDevice            ADMW1001 device context handle
 * @param[in] pDiagnosticsConfig Diagnostics configuration details
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetDiagnosticsConfig(
    ADMW_DEVICE_HANDLE          hDevice,
    ADMW1001_DIAGNOSTICS_CONFIG *pDiagnosticsConfig);

/*!
 * @brief Update channel configuration settings for a specific channel.
 *
 * @param[in] hDevice        ADMW1001 device context handle
 * @param[in] eChannelId     Selects the channel to be updated
 * @param[in] pChannelConfig Channel configuration details
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *          Allows individual channel configuration details to be dynamically
 *          adjusted without rewriting the full device configuration.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetChannelConfig(
    ADMW_DEVICE_HANDLE      hDevice,
    ADMW1001_CH_ID          eChannelId,
    ADMW1001_CHANNEL_CONFIG *pChannelConfig);

/*!
 * @brief Update number of measurements-per-cycle for a specific channel.
 *
 * @param[in] hDevice               ADMW1001 device context handle
 * @param[in] eChannelId            Selects the channel to be updated
 * @param[in] nMeasurementsPerCycle Specifies the number of measurements to be
 *                                  obtained from this channel in each
 *                                  measurement cycle.  Set as 0 to disable the
 *                                  channel (omit from measurement cycle).
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *          Allows individual channels to be dynamically enabled/disabled, and
 *          measurements-per-cycle to be adjusted.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetChannelCount(
    ADMW_DEVICE_HANDLE  hDevice,
    ADMW1001_CH_ID eChannelId,
    uint32_t            nMeasurementsPerCycle);

/*!
 * @brief Update priority level for a specific channel.
 *
 * @param[in] hDevice    ADMW1001 device context handle
 * @param[in] eChannelId Selects the channel to be updated
 * @param[in] ePriority  Specifies the channel priority level
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *          Allows individual channels to be dynamically re-prioritised.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetChannelPriority(
    ADMW_DEVICE_HANDLE          hDevice,
    ADMW1001_CH_ID      eChannelId,
    ADMW1001_CHANNEL_PRIORITY   ePriority);

/*!
 * @brief Update the measurement threshold limits for a specified channel.
 *
 * @param[in] hDevice             ADMW1001 device context handle
 * @param[in] eChannelId          Selects the channel to be updated
 * @param[in] fHighThresholdLimit Optional maximum threshold value for each
 *                                processed sample, to be checked prior to
 *                                publishing.  A channel ALERT condition is
 *                                raised if the processed value is higher than
 *                                this threshold.  Set to NaN if not required.
 * @param[in] fLowThresholdLimit  Optional minimum threshold value for each
 *                                processed sample, to be checked prior to
 *                                publishing.  A channel ALERT condition is
 *                                raised if the processed value is lower than
 *                                this threshold.  Set to NaN if not required.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *          Allows individual channel thresholds to be dynamically adjusted.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetChannelThresholdLimits(
    ADMW_DEVICE_HANDLE      hDevice,
    ADMW1001_CH_ID  eChannelId,
    float32_t               fHighThresholdLimit,
    float32_t               fLowThresholdLimit);


/*!
 * @brief Set a sensor specific parameter for a specified channel.
 *
 * @param[in] hDevice             ADMW1001 device context handle
 * @param[in] eChannelId          Selects the channel to be updated
 * @param[in] fSensorParam        Sensor specific parameter
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *          Allows optional sensor-specific parameter to be specified
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetSensorParameter(
     ADMW_DEVICE_HANDLE     hDevice,
     ADMW1001_CH_ID     eChannelId,
     float32_t              fSensorParam);
/*!
 * @brief Update the extra settling time for a specified channel.
 *
 * @param[in] hDevice       ADMW1001 device context handle
 * @param[in] eChannelId    Selects the channel to be updated
 * @param[in] nSettlingTime A minimum settling time is applied internally for
 *                          each channel, based on the sensor type.  However,
 *                          additional settling time (microseconds) can
 *                          optionally be specified here.  Set to 0 if not
 *                          required.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *          Allows individual channel settling times to be dynamically adjusted.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetChannelSettlingTime(
    ADMW_DEVICE_HANDLE  hDevice,
    ADMW1001_CH_ID eChannelId,
    uint32_t            nSettlingTime);

#ifdef __V2_3_CFG_FMT__
/*!
 * @brief Enable access to advanced sensor configuration options.
 *
 * @param[in] hDevice       ADMW1001 device context handle
 * @param[in] key           Key to unlock advanced access
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details When the correct access key is provided, access to advanced sensor
 *          configuration options and use of advanced sensor types is enabled.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetAdvancedAccess(
    ADMW_DEVICE_HANDLE              hDevice,
    ADMW1001_ADVANCED_ACCESS_KEY    key);
#endif

/*!
 * @brief Assemble a list of separate Look-Up Tables into a single buffer
 *
 * @param[out] pLutBuffer    Pointer to the Look-Up Table data buffer where
 *                           the assembled Look-Up Table data will be placed
 * @param[in] nLutBufferSize Allocated size, in bytes, of the output data buffer
 * @param[in] nNumTables     Number of tables to add to the Look-Up Table buffer
 * @param[in] ppDesc         Array of pointers to the table descriptors to be added
 * @param[in] ppData         Array of pointers to the table data to be added
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details This utiliity function fills the Look-up Table header fields; then
 *          walks through the array of individual table descriptor and data
 *          pointers provided, appending (copying) each one to the Look-Up Table
 *          data buffer. The length and crc16 fields of each table descriptor
 *          will be calculated and filled by this function, but other fields in
 *          the descriptor structure must be filled by the caller beforehand.
 *
 * @note    The assembled LUT data buffer filled by this function can then be
 *          written to the device memory using @ref admw1001_SetLutData.
 */
ADMW_RESULT admw1001_AssembleLutData(
    ADMW1001_LUT            *pLutBuffer,
    unsigned                nLutBufferSize,
    unsigned                const nNumTables,
    ADMW1001_LUT_DESCRIPTOR *const ppDesc[],
    ADMW1001_LUT_TABLE_DATA *const ppData[]);

/*!
 * @brief Write Look-Up Table data to the device memory
 *
 * @param[in]  hDevice  ADMW1001 device context handle
 * @param[out] pLutData Pointer to the Look-Up Table data structure
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Validates the Look-Up Table data format and loads it into
 *          device memory via dedicated keyhole registers.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetLutData(
    ADMW_DEVICE_HANDLE  hDevice,
    ADMW1001_LUT        *const pLutData);

/*!
 * @brief Write Look-Up Table raw data to the device memory
 *
 * @param[in]  hDevice  ADMW device context handle
 * @param[out] pLutData Pointer to the Look-Up Table raw data structure
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details This can be used instead of @ref admw1001_SetLutData for
 *          loading LUT data from the alternative raw data format.  See
 *          @ref admw1001_SetLutData for more information.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetLutDataRaw(
    ADMW_DEVICE_HANDLE  hDevice,
    ADMW1001_LUT_RAW    *const pLutData);

/*!
 * @brief Get the number of samples available when DATAREADY status is asserted.
 *
 * @param[in]  hDevice               ADMW device context handle
 * @param[in]  eMeasurementMode      Must be set to the same value used for @ref
 *                                   admw_StartMeasurement().
 * @param[out] peOperatingMode       Pointer to return the configured operating mode
 * @param[out] peDataReadyMode       Pointer to return the configured data publishing mode
 * @param[out] pnSamplesPerDataready Pointer to return the calculated number of samples
 *                                   available when DATAREADY is asserted
 * @param[out] pnSamplesPerCycle     Pointer to return the calculated number of samples
 *                                   produced per measurement cycle
 * @param[out] pnBytesPerSample      Pointer to return the size, in bytes, of each sample
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Examines the current configuration settings in the device registers
 *          to calculate the number of samples available whenever the DATAREADY
 *          signal is asserted, along with other related information.  This may
 *          be used to allocate buffers to store samples and to determine how
 *          many samples to retrieve whenever the DATAREADY status is asserted.
 */
ADMW_RESULT admw1001_GetDataReadyModeInfo(
    ADMW_DEVICE_HANDLE      const hDevice,
    ADMW_MEASUREMENT_MODE   const eMeasurementMode,
    ADMW1001_OPERATING_MODE *const peOperatingMode,
    ADMW1001_DATAREADY_MODE *const peDataReadyMode,
    uint32_t                *const pnSamplesPerDataready,
    uint32_t                *const pnSamplesPerCycle,
    uint8_t                 *const pnBytesPerSample);

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADMW1001_API_H__ */
