/******************************************************************************
Copyright 2017 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *****************************************************************************/

/*!
 ******************************************************************************
 * @file:
 * @brief:  ADMW OS-dependent wrapper layer for SPI interface
 *-----------------------------------------------------------------------------
 */

#include <mbed.h>

#include "inc/admw_spi.h"
#include "inc/admw_log.h"

#define ADMW_SPI_MODE       0 /* CPOL=0, CPHA=0 */
#define ADMW_SPI_FRAME_SIZE 8 /* 8-bit frame size */


#ifdef __cplusplus
extern "C" {
#endif


// Struct to contain anything needed to identify the SPI device
// This is returned as a ADMW_SPI_HANDLE*, and is required to use
// the other SPI functions
typedef struct {
    SPI *_spi;
    DigitalOut *_cs;
    DigitalOut *_wakeup;
} SpiContext_t;


/*
 * Open the SPI interface and allocate resources
 */
ADMW_RESULT admw_SpiOpen(
    ADMW_PLATFORM_SPI_CONFIG *pConfig,
    ADMW_SPI_HANDLE *phDevice)
{
    SpiContext_t *pCtx = (SpiContext_t*)malloc(sizeof(*pCtx));
    if(!pCtx) {
        ADMW_LOG_ERROR("Failed to allocate memory for SPI device");
        return ADMW_NO_MEM;
    }
    
    pCtx->_spi = new SPI((PinName)pConfig->mosiPin,
                (PinName)pConfig->misoPin,
                (PinName)pConfig->sckPin);
    pCtx->_cs = new DigitalOut((PinName)pConfig->csPin, 1);
    
    pCtx->_spi->format(ADMW_SPI_FRAME_SIZE, ADMW_SPI_MODE);
    pCtx->_spi->frequency(pConfig->maxSpeedHz);
    
    pCtx->_wakeup = new DigitalOut((PinName)pConfig->wakeupPin, 1);
    
    *phDevice = (ADMW_SPI_HANDLE)pCtx;
    
    return ADMW_SUCCESS;
}

ADMW_RESULT
admw_SpiReceive(
    ADMW_SPI_HANDLE hDevice,
    void *pTxData,
    void *pRxData,
    unsigned nLength,
    bool bCsHold)
{
    SpiContext_t *pCtx = (SpiContext_t*)hDevice;
    
    pCtx->_spi->format(ADMW_SPI_FRAME_SIZE, 1);
    
    int rc = 0;
   
    *(pCtx->_wakeup) = 1;
    wait_us(31);
    *(pCtx->_cs) = 0;
     
    rc  = pCtx->_spi->write((char*)(pTxData), pTxData ? nLength : 0,
                     (char*)(pRxData), pRxData ? nLength : 0);
    
    if ((rc < 0) || !bCsHold)
    {
       *(pCtx->_cs) = 1;
    }
    *(pCtx->_wakeup) = 0;
       
    if (rc < 0)
    {
        ADMW_LOG_ERROR("Failed to complete SPI transfer");
        return ADMW_FAILURE;
    }
    
    pCtx->_spi->format(ADMW_SPI_FRAME_SIZE, ADMW_SPI_MODE);
    
    return ADMW_SUCCESS;
}

/*
 * Execute a bi-directional data transfer on the SPI interface
 */
ADMW_RESULT
admw_SpiTransfer(
    ADMW_SPI_HANDLE hDevice,
    void *pTxData,
    void *pRxData,
    unsigned nLength,
    bool bCsHold)
{
    int rc = 0;
    
    SpiContext_t *pCtx = (SpiContext_t*)hDevice;

    *(pCtx->_wakeup) = 1;
    wait_us(31);
    *(pCtx->_cs) = 0;
    *(pCtx->_wakeup) = 0;
    rc  = pCtx->_spi->write((char*)(pTxData), pTxData ? nLength : 0,
                     (char*)(pRxData), pRxData ? nLength : 0);
    
    if ((rc < 0) || !bCsHold)
        *(pCtx->_cs) = 1;

    if (rc < 0)
    {
        ADMW_LOG_ERROR("Failed to complete SPI transfer");
        return ADMW_FAILURE;
    }
    
    return ADMW_SUCCESS;
}

/*
 * Close the SPI interface and free resources
 */
void admw_SpiClose(
    ADMW_SPI_HANDLE hDevice)
{
    SpiContext_t *pCtx = (SpiContext_t *)hDevice;

    delete pCtx->_spi;
    delete pCtx->_cs;
    delete pCtx->_wakeup;
    
    free(pCtx);
}

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */
