/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:   admw_api.h
 * @brief:  ADMW Host Library Application Programming Interface (API)
 *-----------------------------------------------------------------------------
 */

#ifndef _ADMW_API_H__
#define _ADMW_API_H__

#include "inc/admw_types.h"
#include "inc/admw_config_types.h"
#include "inc/admw_platform.h"
#include "inc/admw_gpio.h"
#include "inc/admw_spi.h"
#include "inc/admw_log.h"
#include "inc/admw_time.h"
#include "inc/admw1001/admw1001_sensor_types.h"
#include "inc/admw1001/admw1001_lut_data.h"
#include "ADMW1001_REGISTERS_typedefs.h"
/*! @defgroup ADMW_Api ADMW Host Library API
 *  Host library API common to the ADMW product family.
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! A handle used in all API functions to identify the ADMW device. */
typedef void* ADMW_DEVICE_HANDLE;

#define SPI_BUFFER_SIZE                        100u

#define SFL_READ_STATUS_HDR_SIZE               14u

#define SFL_READ_STATUS_RESPONSE_SIZE          42u

/*! Supported connection types for communication with the ADMW device. */
typedef enum
{
    ADMW_CONNECTION_TYPE_SPI = 1,
    /*!< Serial Peripheral Interface (SPI) connection type */

}   ADMW_CONNECTION_TYPE;

/*! Connection details for communication with a ADMW device instance. */
typedef struct
{
    ADMW_CONNECTION_TYPE      type;
    /*!< Connection type selection */
    ADMW_PLATFORM_SPI_CONFIG  spi;
    /*!< SPI connection parameters, required if SPI connection type is used */
    ADMW_PLATFORM_GPIO_CONFIG gpio;
    /*!< GPIO connection parameters, for device reset and status I/O signals */
    ADMW_PLATFORM_LOG_CONFIG  log;
    /*!< Log interface connection parameters, for display/routing of log messages */

}   ADMW_CONNECTION;

/*! Bit masks (flags) for the different device status indicators. */
typedef enum
{
    ADMW_DEVICE_STATUS_BUSY         = (1 << 0),
    /*!< Indicates that a command is currently running on the device */
    ADMW_DEVICE_STATUS_DATAREADY    = (1 << 1),
    /*!< Indicates the availability of measurement data for retrieval */
    ADMW_DEVICE_STATUS_ERROR        = (1 << 2),
    /*!< Indicates that an error condition has been detected by the device */
    ADMW_DEVICE_STATUS_ALERT        = (1 << 3),
    /*!< Indicates that an alert condition has been detected by the device */
    ADMW_DEVICE_STATUS_FIFO_ERROR   = (1 << 4),
    /*!< Indicates that a FIFO error condition has been detected by the device */
    ADMW_DEVICE_STATUS_CONFIG_ERROR = (1 << 5),
    /*!< Indicates that a configuration error condition has been detected by the device */
    ADMW_DEVICE_STATUS_LUT_ERROR    = (1 << 6),
    /*!< Indicates that a look-up table error condition has been detected by the device */

}   ADMW_DEVICE_STATUS_FLAGS;


/*! Bit masks (flags) for the different channel alert indicators. */
typedef enum
{
    ADMW_ALERT_DETAIL_CH_RESULT_VALID             = (1 << 0),
    /*!< Indicates valid result in the channel */
    ADMW_ALERT_DETAIL_CH_ADC_NEAR_OVERRANGE       = (1 << 1),
    /*!< Indicates ADC is near overrange in the channel */
    ADMW_ALERT_DETAIL_CH_SENSOR_UNDERRANGE        = (1 << 2),
    /*!< Indicates sensor underrange in the channel */
    ADMW_ALERT_DETAIL_CH_SENSOR_OVERRANGE         = (1 << 3),
    /*!< Indicates sensor overrange in the channel */
    ADMW_ALERT_DETAIL_CH_CJ_SOFT_FAULT            = (1 << 4),
    /*!< Indicates Cold Junction soft fault in the channel */
    ADMW_ALERT_DETAIL_CH_CJ_HARD_FAULT            = (1 << 5),
    /*!< Indicates Cold Junction Hard Fault in the channel */
    ADMW_ALERT_DETAIL_CH_ADC_INPUT_OVERRANGE      = (1 << 6),
    /*!< Indicates ADC Input overrange in the channel */
    ADMW_ALERT_DETAIL_CH_SENSOR_HARDFAULT         = (1 << 7),
    /*!< Indicates Hard Fault in the channel */
}   ADMW_CHANNEL_ALERT_FLAGS;

/*! Status details retreived from the ADMW device. */
typedef struct
{
    ADMW_DEVICE_STATUS_FLAGS      deviceStatus;
    /*!< General summary status information from the device  */
    ADMW_CHANNEL_ALERT_FLAGS      channelAlerts[ADMW1001_MAX_CHANNELS];
    /*!< Per-channel alert status information from the device  */
    uint32_t                      errorCode;
    /*!< Code identifying the last error signalled by the device  */
    uint32_t                      alertCode;
    /*!< Code identifying the last alert signalled by the device  */
    uint32_t                      debugCode;
    /*!< Supplementary code related to the last error/alert  */
    uint32_t                      channelAlertCodes[ADMW1001_MAX_CHANNELS];
    /*!< Per-channel code identifying the last alert signalled for each channel */

}   ADMW_STATUS;

/*! Data sample details retreived from the ADMW device. */
typedef struct
{
    float32_t                     processedValue;
    /*!< The processed value obtained from the measurement channel, as a final
     *   measurement value, following calibration and linearisation correction,
     *   and conversion into an appropriate unit of measurement.
     */
    uint16_t    Channel_ID :  4;    /**< Indicates which channel this result corresponds to */
    uint16_t    Ch_Error   :  1;    /**< Indicates Error on channel */
    uint16_t    Ch_Alert   :  1;    /**< Indicates Alert on channel */
    uint16_t    Ch_Raw     :  1;    /**< Indicates if Raw sensor data field is valid */
    uint16_t    Ch_Valid   :  1;    /**< Indicates if this Result structure is valid */
    uint16_t    Reserved   :  8;   /**< Reserved for future use */
    ADMW_CORE_Alert_Detail_Ch_t    Status;      /**< Indicates Status of this measurement */
    float32_t   Raw_Sample;
    /*!< The raw value obtained from the measurement channel, before
     *   calibration and linearisation correction, and conversion.
     */

}   ADMW_DATA_SAMPLE;

/*! Measurement mode options for the ADMW device.
 *  @ref admw_StartMeasurement
 */
typedef enum
{
    ADMW_MEASUREMENT_MODE_NORMAL = 0,
    /*!< In this mode, normal measurement cycle(s) are executed and data samples
     *   are returned with raw measurement values included. */
    ADMW_MEASUREMENT_MODE_OMIT_RAW,
    /*!< In this mode, normal measurement cycle(s) are executed and data samples
     *   are returned with raw measurement values omitted for efficiency. */

}   ADMW_MEASUREMENT_MODE;

/*! Identifiers for the user configuration slots in persistent memory. */
typedef enum
{
    ADMW_FLASH_CONFIG_1,

}   ADMW_USER_CONFIG_SLOT;

/*! Identifiers for the user configuration slots in persistent memory. */
typedef enum
{
    ADMW_FLASH_LUT_CHANNEL_0=0,
    ADMW_FLASH_LUT_CHANNEL_1,
    ADMW_FLASH_LUT_CHANNEL_2,
    ADMW_FLASH_LUT_CHANNEL_3,
    

}   ADMW_USER_LUT_CONFIG_SLOT;

typedef struct
{
    unsigned nDeviceIndex;
    ADMW_SPI_HANDLE hSpi;
    ADMW_GPIO_HANDLE hGpio;
} ADMW_DEVICE_CONTEXT;

/******************************************************************************
 * ADMW High-Level API function prototypes
 *****************************************************************************/

/*!
 * @brief Open ADMW device handle and set up communication interface.
 *
 * @param[in]  nDeviceIndex    Zero-based index number identifying this device
 *                             instance.  Note that this will be used to
 *                             retrieve a specific device configuration for
 *                             this device (see @ref admw_SetConfig
 *                             and @ref ADMW_CONFIG)
 * @param[in]  pConnectionInfo Host-specific connection details (e.g. SPI, GPIO)
 * @param[out] phDevice        Pointer to return an ADMW device handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *         - #ADMW_NO_MEM  Failed to allocate memory resources.
 *         - #ADMW_INVALID_DEVICE_NUM Invalid device index specified
 *
 * @details Configure and initialise the Log interface and the SPI/GPIO
 *          communication interface to the ADISense module.
 */
ADMW_RESULT admw_Open(
    unsigned                   const nDeviceIndex,
    ADMW_CONNECTION     * const pConnectionInfo,
    ADMW_DEVICE_HANDLE  * const phDevice);

/*!
 * @brief Close ADMW device context and free resources.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 */
ADMW_RESULT admw_Close(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Get the current state of the specified GPIO input signal.
 *
 * @param[in]  hDevice    ADMW device context handle
 * @param[in]  ePinId     GPIO pin to query
 * @param[out] pbAsserted Pointer to return the state of the status signal GPIO pin
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *         - #ADMW_INVALID_DEVICE_NUM Invalid GPIO pin specified.
 *
 * @details Sets *pbAsserted to true if the status signal is asserted, or false
 *          otherwise.
 */
ADMW_RESULT admw_GetGpioState(
    ADMW_DEVICE_HANDLE   const hDevice,
    ADMW_GPIO_PIN        const ePinId,
    bool                  * const pbAsserted);

/*!
 * @brief Register an application-defined callback function for GPIO interrupts
 *
 * @param[in] hDevice          ADMW context handle (@ref admw_Open)
 * @param[in] ePinId           GPIO pin on which to enable/disable interrupts
 * @param[in] callbackFunction Function to be called when an interrupt occurs.
 *                             Specify NULL here to disable interrupts.
 * @param[in] pCallbackParam   Optional opaque parameter passed to the callback
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *         - #ADMW_INVALID_DEVICE_NUM Invalid GPIO pin specified.
 */
ADMW_RESULT admw_RegisterGpioCallback(
    ADMW_DEVICE_HANDLE          const hDevice,
    ADMW_GPIO_PIN               const ePinId,
    ADMW_GPIO_CALLBACK          const callbackFunction,
    void                           * const pCallbackParam);

/*!
 * @brief Trigger a shut down of the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to initiate a shut down,
 *          typically used to conserve power when the device is not
 *          in use.  The device may be restarted by calling
 *          @ref admw_Reset().  Note that active configuration
 *          settings are not preserved during shutdown and must be
 *          reloaded after the device has become ready again.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_Shutdown(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Reset the ADMW device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Trigger a hardware-reset of the ADMW device.
 *
 * @note The device may require several seconds before it is ready for use
 *       again. @ref admw_GetDeviceReadyState may be used to check if
 *       the device is ready.
 */
ADMW_RESULT admw_Reset(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Check if the device is ready, following power-up or a reset.
 *
 * @param[in]  hDevice ADMW device context handle
 * @param[out] pbReady Pointer to return true if the device is ready, or false
 *                     otherwise
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details This function attempts to read a fixed-value device register via
 *           the communication interface.
 */
ADMW_RESULT admw_GetDeviceReadyState(
    ADMW_DEVICE_HANDLE    const hDevice,
    bool                   * const pbReady);

/*!
 * @brief Obtain the product ID from the device.
 *
 * @param[in]  hDevice    ADMW device context handle
 * @param[out] pProductId Pointer to return the product ID value
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the product ID registers on the device and returns the value.
 */
ADMW_RESULT admw_GetProductID(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_PRODUCT_ID     * const pProductId);

/*!
 * @brief Write full configuration settings to the device registers.
 *
 * @param[in]  hDevice ADMW device context handle
 * @param[out] pConfig Pointer to the configuration data structure
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Translates configuration details provided into device-specific
 *          register settings and updates device configuration registers.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw_SetConfig(
    ADMW_DEVICE_HANDLE   const hDevice,
    ADMW_CONFIG        * const pConfig);

/*!
 * @brief Apply the configuration settings currently stored in device registers
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to reload and apply configuration
 *          from the device configuration registers.  Changes to configuration
 *          registers are ignored by the device until this function is called.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_ApplyConfigUpdates(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Store the configuration settings to persistent memory on the device.
 *
 * @param[in] hDevice ADMW device context handle
 * @param[in] eSlotId User configuration slot in persistent memory
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to save the current contents of its
 *          device configuration registers to non-volatile memory.
 *
 * @note No other command must be running when this is called.
 * @note Do not power down the device while this command is running.
 */
ADMW_RESULT admw_SaveConfig(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_USER_CONFIG_SLOT const eSlotId);

/*!
 * @brief Restore configuration settings from persistent memory on the device.
 *
 * @param[in] hDevice ADMW device context handle
 * @param[in] eSlotId User configuration slot in persistent memory
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to restore the contents of its
 *          device configuration registers from non-volatile memory.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RestoreConfig(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_USER_CONFIG_SLOT const eSlotId);

/*!
 * @brief Store the LUT data to persistent memory on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to save the current contents of its
 *          LUT data buffer, set using @ref admw_SetLutData, to
 *          non-volatile memory.
 *
 * @note No other command must be running when this is called.
 * @note Do not power down the device while this command is running.
 */
ADMW_RESULT admw_SaveLutData(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Restore LUT data from persistent memory on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to restore the contents of its
 *          LUT data, previously stored with @ref admw_SaveLutData, from
 *          non-volatile memory.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RestoreLutData(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Start the measurement cycles on the device.
 *
 * @param[in] hDevice          ADMW device context handle
 * @param[in] eMeasurementMode Allows a choice of special modes for the
 *                             measurement.  See @ref ADMW_MEASUREMENT_MODE
 *                             for further information.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to start executing measurement cycles
 *          according to the current applied configuration settings.  The
 *          DATAREADY status signal will be asserted whenever new measurement
 *          data is published, according to selected settings.
 *          Measurement cycles may be stopped by calling @ref
 *          admw_StopMeasurement.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_StartMeasurement(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_MEASUREMENT_MODE const eMeasurementMode);

/*!
 * @brief Stop the measurement cycles on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to stop executing measurement cycles.
 *          The command may be delayed until the current conversion, if any, has
 *          been completed and published.
 *
 * @note To be used only if a measurement command is currently running.
 */
ADMW_RESULT admw_StopMeasurement(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in diagnostic checks on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to execute its built-in diagnostic
 *          tests, on any enabled measurement channels, according to the current
 *          applied configuration settings.  Device status registers will be
 *          updated to indicate if any errors were detected by the diagnostics.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RunDiagnostics(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in calibration on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to execute its self-calibration
 *          routines, on any enabled measurement channels, according to the
 *          current applied configuration settings.  Device status registers
 *          will be updated to indicate if any errors were detected.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RunCalibration(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Run built-in digital calibration on the device.
 *
 * @param[in] hDevice ADMW device context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Instructs the ADMW device to execute its calibration
 *          routines, on any enabled digital channels, according to the
 *          current applied configuration settings.  Device status registers
 *          will be updated to indicate if any errors were detected.
 *
 * @note No other command must be running when this is called.
 */
ADMW_RESULT admw_RunDigitalCalibration(
    ADMW_DEVICE_HANDLE    const hDevice);

/*!
 * @brief Read the current status from the device registers.
 *
 * @param[in]  hDevice ADMW device context handle
 * @param[out] pStatus Pointer to return the status summary obtained from the
 *                     device.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the status registers and extracts the relevant information
 *          to return to the caller.
 *
 * @note This may be called at any time, assuming the device is ready.
 */
ADMW_RESULT admw_GetStatus(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_STATUS         * const pStatus);

/*!
 * @brief Assemble a list of separate Look-Up Tables into a single buffer
 *
 * @param[out] pLutBuffer    Pointer to the Look-Up Table data buffer where
 *                           the assembled Look-Up Table data will be placed
 * @param[in] nLutBufferSize Allocated size, in bytes, of the output data buffer
 * @param[in] nNumTables     Number of tables to add to the Look-Up Table buffer
 * @param[in] ppDesc         Array of pointers to the table descriptors to be added
 * @param[in] ppData         Array of pointers to the table data to be added
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details This utiliity function fills the Look-up Table header fields; then
 *          walks through the array of individual table descriptor and data
 *          pointers provided, appending (copying) each one to the Look-Up Table
 *          data buffer. The length and crc16 fields of each table descriptor
 *          will be calculated and filled by this function, but other fields in
 *          the descriptor structure must be filled by the caller beforehand.
 *
 * @note    The assembled LUT data buffer filled by this function can then be
 *          written to the device memory using @ref admw1001_SetLutData.
 */
ADMW_RESULT admw1001_AssembleLutData(
    ADMW1001_LUT            *pLutBuffer,
    unsigned                nLutBufferSize,
    unsigned                const nNumTables,
    ADMW1001_LUT_DESCRIPTOR *const ppDesc[],
    ADMW1001_LUT_TABLE_DATA *const ppData[]);

/*!
 * @brief Write Look-Up Table data to the device memory
 *
 * @param[in]  hDevice  ADMW1001 device context handle
 * @param[out] pLutData Pointer to the Look-Up Table data structure
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Validates the Look-Up Table data format and loads it into
 *          device memory via dedicated keyhole registers.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetLutData(
    ADMW_DEVICE_HANDLE  hDevice,
    ADMW1001_LUT        *const pLutData);

/*!
 * @brief Write Look-Up Table raw data to the device memory
 *
 * @param[in]  hDevice  ADMW device context handle
 * @param[out] pLutData Pointer to the Look-Up Table raw data structure
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details This can be used instead of @ref admw1001_SetLutData for
 *          loading LUT data from the alternative raw data format.  See
 *          @ref admw1001_SetLutData for more information.
 *
 * @note Settings are not applied until admw_ApplyConfigUpdates() is called
 */
ADMW_RESULT admw1001_SetLutDataRaw(
    ADMW_DEVICE_HANDLE  hDevice,
    ADMW1001_LUT_RAW    *const pLutData);
/*!
 * @brief Read measurement data samples from the device registers.
 *
 * @param[in]  hDevice          ADMW device context handle
 * @param[in]  eMeasurementMode Must be set to the same value used for @ref
 *                              admw_StartMeasurement().
 * @param[out] pSamples         Pointer to return a set of requested data samples.
 * @param[in]  nBytesPerSample  The size, in bytes, of each sample.
 * @param[in]  nRequested       Number of requested data samples.
 * @param[out] pnReturned       Number of valid data samples successfully retrieved.
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the status registers and extracts the relevant information
 *          to return to the caller.
 *
 * @note This is intended to be called only when the DATAREADY status signal
 *       is asserted.
 */
ADMW_RESULT admw_GetData(
    ADMW_DEVICE_HANDLE    const hDevice,
    ADMW_MEASUREMENT_MODE const eMeasurementMode,
    ADMW_DATA_SAMPLE    * const pSamples,
    uint8_t                    const nBytesPerSample,
    uint32_t                   const nRequested,
    uint32_t                 * const pnReturned);

/*!
 * @brief Check if a command is currently running on the device.
 *
 * @param[in]  hDevice          ADMW device context handle
 * @param[out] pbCommandRunning Pointer to return the command running status
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully.
 *
 * @details Reads the device status register to check if a command is running.
 */
ADMW_RESULT admw_GetCommandRunningState(
    ADMW_DEVICE_HANDLE hDevice,
    bool *pbCommandRunning);

ADMW_RESULT admw1001_sendRun( ADMW_DEVICE_HANDLE   const hDevice);
ADMW_RESULT admw_deviceInformation(ADMW_DEVICE_HANDLE hDevice);

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* _ADMW_API_H__ */
