/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:
 * @brief:  Look-Up Table data-type definitions for ADMW1001 API.
 *-----------------------------------------------------------------------------
 */

#ifndef __ADMW1001_LUT_DATA_H__
#define __ADMW1001_LUT_DATA_H__

#include "admw_types.h"
#include "admw1001_sensor_types.h"

//lint --e{38} suppress "offset of symbol"

/*! @addtogroup ADMW1001_Api
 *  @{
 */

#ifdef __cplusplus
extern "C" {
#endif

/*! LUT data validation signature */
#define ADMW_LUT_SIGNATURE  0x4C555473

/*! LUT data CRC-16-CCITT seed value */
#define ADMW_LUT_CRC_SEED   0x4153

/*! LUT maximum allowed size  */
#define ADMW_LUT_MAX_SIZE 10240U

#define MAX_LUT_NUM_ENTRIES 16
/*! Linearisation look-up table / co-efficient list geometry */
typedef enum {
    ADMW1001_LUT_GEOMETRY_RESERVED = 0x00,
    /**< reserved - for internal use only */
    ADMW1001_LUT_GEOMETRY_COEFFS = 0x01,
    /**< 1D/2D equation coefficient list */
    ADMW1001_LUT_GEOMETRY_NES_1D = 0x02,
    /**< 1-dimensional not-equally-spaced look-up table */
} ADMW1001_LUT_GEOMETRY;

/*! Linearisation equation type */
typedef enum {
    ADMW1001_LUT_EQUATION_POLYN,
    /**< Polynomial equation, typically used for Thermocouple and RTD
     * linearisation */
    ADMW1001_LUT_EQUATION_POLYNEXP,
    /**< Polynomial + exponential equation, typically used for Thermocouple
     * inverse linearisation */
    ADMW1001_LUT_EQUATION_QUADRATIC,
    /**< Quadratic linearisation equation, typically used for RTD
     * linearisation */
    ADMW1001_LUT_EQUATION_STEINHART,
    /**< Steinhart-Hart equation, typically used for Thermistor
     * linearisation */
    ADMW1001_LUT_EQUATION_LOGARITHMIC,
    /**< Beta-based logarithmic equation, typically used for Thermistor
     * linearisation */
    ADMW1001_LUT_EQUATION_BIVARIATE_POLYN,
    /**< Bi-variate polynomial equation, typically used for bridge pressure
     * sensor linearisation
     * @note 2nd-degree is the maximum currently supported
     */
    ADMW1001_LUT_EQUATION_COUNT,
    /**< Enum count value - for internal use only */
    ADMW1001_LUT_EQUATION_LUT,
    /**< Hard-coded Look-Up Table - for internal use only */
} ADMW1001_LUT_EQUATION;

typedef enum {
    ADMW1001_LUT_TC_DIRECTION_FORWARD,
    /**< Thermocouple forward (mV to Celsius) linearisation
     *   Use this value by default for non-thermocouple sensors */
    ADMW1001_LUT_TC_DIRECTION_BACKWARD,
    /**< Thermocouple inverse (Celsius to mV) linearisation */
    ADMW1001_LUT_TC_DIRECTION_COUNT,
    /**< Enum count value - for internal use only */
} ADMW1001_LUT_TC_DIRECTION;

/*! Linearisation data vector format */
typedef enum {
    ADMW1001_LUT_DATA_TYPE_RESERVED = 0,
    /**< Reserved - for internal use only */
    ADMW1001_LUT_DATA_TYPE_FLOAT32 = 1,
    /**< Single-precision 32-bit floating-point */
    ADMW1001_LUT_DATA_TYPE_FLOAT64 = 2,
    /**< Double-precision 64-bit floating-point */
} ADMW1001_LUT_DATA_TYPE;

/*! Struct for a list of coefficients to be used in an equation */
typedef struct __attribute__((packed, aligned(4))){
    uint32_t nCoeffs;
    /**< number of coefficients */
    float32_t rangeMin;
    /**< look-up table range - minimum */
    float32_t rangeMax;
    /**< look-up table range - maximum */
    float64_t coeffs[];
    /**< C99 flexible array: sorted by ascending exponent in polynomials */
} ADMW1001_LUT_COEFF_LIST;

/*! Struct for a 1-dimensional equally-spaced look-up table */
typedef struct __attribute__((packed, aligned(4))){
    uint32_t nElements;
    /**< number of elements. */
    float32_t initInputValue;
    /**< initial input value, corresponding to first table element */
    float32_t inputValueIncrement;
    /**< interval between successive input values */
    float32_t lut[];
    /**< C99 flexible array */
} ADMW1001_LUT_1D_ES;

/*! Struct for a 1-dimensional not-equally-spaced look-up table */
typedef struct __attribute__((packed, aligned(4))){
    uint32_t nElements;
    /**< number of elements of each array. */
    float32_t lut[];
    /**< C99 flexible array, first X's array then Y's array*/
} ADMW1001_LUT_1D_NES;

/*! Macro to calculate the number of elements in
 * a @ref ADMW1001_LUT_1D_NES table */
#define ADMW1001_LUT_1D_NES_NELEMENTS(_t) \
        ((_t).nElements * 2)

/*! Macro to calculate the storage size in bytes of
 * a @ref ADMW1001_LUT_1D_NES table */
#define ADMW1001_LUT_1D_NES_SIZE(_t)                              \
    (sizeof(_t) + (sizeof(float32_t) * ADMW1001_LUT_1D_NES_NELEMENTS(_t)))

/*! Look-Up Table descriptor */
typedef union __attribute__((packed, aligned(4))) {
    struct {
        uint16_t geometry   : 4;    /**< ADMW1001_LUT_GEOMETRY */
        uint8_t  channel    : 2;    /**< ADMW1001_ADC_CHANNEL */
        uint16_t equation   : 6;    /**< ADMW1001_LUT_EQUATION */
        uint16_t dir        : 4;    /**< ADMW1001_LUT_TC_DIRECTION */
        uint16_t sensor     : 12;   /**< ADMW1001_ADC_SENSOR_TYPE */
        uint16_t dataType   : 4;    /**< ADMW1001_LUT_DATA_TYPE */
        uint16_t length;            /**< Length (bytes) of table data section
                                        (excl. this header) */
    };
} ADMW1001_LUT_DESCRIPTOR;

/*! Look-Up Table geometry-specific data structures */
typedef union {
    ADMW1001_LUT_COEFF_LIST          coeffList;
    /**< Data format for tables with ADMW1001_LUT_GEOMETRY_COEFFS geometry
     *   except where equation is ADMW1001_LUT_EQUATION_BIVARIATE_POLYN */
    ADMW1001_LUT_1D_NES              lut1dNes;
    /**< Data format for tables with ADMW1001_LUT_GEOMETRY_NES_1D geometry */
} ADMW1001_LUT_TABLE_DATA;

/*! Look-Up Table structure */
typedef struct __attribute__((packed, aligned(4))) {
    ADMW1001_LUT_DESCRIPTOR descriptor;
    /**< Look-Up Table descriptor */
    ADMW1001_LUT_TABLE_DATA data;
    /**< Look-Up Table data */
} ADMW1001_LUT_TABLE;

/*! LUT data format versions */
typedef struct __attribute__((packed, aligned(4))) {
    uint8_t major; /*!< Major version number */
    uint8_t minor; /*!< Minor version number */
} ADMW1001_LUT_VERSION;

/*! LUT data header structure */
typedef struct __attribute__((packed, aligned(4))) {
    uint32_t signature;
    /**< Hard-coded signature value (@ref ADMW_LUT_SIGNATURE) */
    ADMW1001_LUT_VERSION version;
    /**< LUT data format version (@ref ADMW_LUT_VERSION) */
    uint16_t numTables;
    /**< Total number of tables */
    uint32_t totalLength;
    /**< Total length (in bytes) of all table descriptors and data
     *   (excluding this header)
     *   This, plus the header length, must not exceed ADMW_LUT_MAX_SIZE
     */
} ADMW1001_LUT_HEADER;

/*! LUT data top-level structure */
typedef struct __attribute__((packed, aligned(4))) {
    ADMW1001_LUT_HEADER header;
    /*!< LUT data top-level header structure */
    ADMW1001_LUT_TABLE tables[1];
        /*!< Variable-length array of one-or-more look-up table structures */
} ADMW1001_LUT;

/*! Alternative top-level structure for raw LUT data representation
 *
 * @note This is intended to be used for encapsulating the storage of static
 *       LUT data declarations in C files.  The rawTableData can be cast
 *       to the ADMW_LUT type for further parsing/processing.
 */
typedef struct __attribute__((packed, aligned(4))) {
    ADMW1001_LUT_HEADER header;
    /*!< LUT data top-level header structure */
    uint8_t rawTableData[];
    /*!< Variable-length byte array of look-up tables in raw binary format */
} ADMW1001_LUT_RAW;

#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADMW1001_LUT_DATA_H__ */