/*
Copyright 2019 (c) Analog Devices, Inc.

All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
  - Redistributions of source code must retain the above copyright
    notice, this list of conditions and the following disclaimer.
  - Redistributions in binary form must reproduce the above copyright
    notice, this list of conditions and the following disclaimer in
    the documentation and/or other materials provided with the
    distribution.
  - Neither the name of Analog Devices, Inc. nor the names of its
    contributors may be used to endorse or promote products derived
    from this software without specific prior written permission.
  - The use of this software may or may not infringe the patent rights
    of one or more patent holders. This license does not release you
    from the requirement that you obtain separate licenses from these
    patent holders to use this software.
  - Use of the software either in source or binary form, must be run
    on or directly connected to an Analog Devices Inc. component.

THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

/*!
 ******************************************************************************
 * @file:   admw_spi.h
 * @brief:  ADMW OS-dependent wrapper layer for SPI interface
 *-----------------------------------------------------------------------------
 */

#ifndef __ADMW_SPI_H__
#define __ADMW_SPI_H__

#include "inc/admw_types.h"
#include "inc/admw_platform.h"

/*! @ingroup ADMW_Host */

/*! @addtogroup ADMW_Spi ADMW Host SPI interface functions
 *  @{
 */

/*! A handle used in all API functions to identify the SPI interface context */
typedef void * ADMW_SPI_HANDLE;

#ifdef __cplusplus
extern "C"
{
#endif

/*!
 * @brief Open the SPI interface and allocate resources
 *
 * @param[in]  pConfig  Pointer to platform-specific SPI interface details
 * @param[out] phDevice Pointer to return a SPI interface context handle
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully
 *         - #ADMW_NO_MEM  Failed to allocate memory for interface context
 */
ADMW_RESULT admw_SpiOpen(
    ADMW_PLATFORM_SPI_CONFIG * pConfig,
    ADMW_SPI_HANDLE          * phDevice);

/*!
 * @brief Close SPI interface and free resources
 *
 * @param[in] hDevice SPI interface context handle (@ref admw_SpiOpen)
 */
void admw_SpiClose(
    ADMW_SPI_HANDLE            hDevice);

/*!
 * @brief Transfer data to slave device
 *
 * @param[in] hDevice SPI interface context handle (@ref admw_SpiOpen)
 * @param[in] pTxData Transmit data buffer, or NULL for read-only transfers
 * @param[in] pRxData Receive  data buffer, or NULL for write-only transfers
 * @param[in] nLength Number of bytes to transfer
 * @param[in] bCsHold Leave the chip-select asserted when the transfer is done
 *
 * @return Status
 *         - #ADMW_SUCCESS Call completed successfully
 *         - #ADMW_FAILURE Failed to complete SPI transfer
 */
ADMW_RESULT admw_SpiTransfer(
    ADMW_SPI_HANDLE            hDevice,
    void                          * pTxData,
    void                          * pRxData,
    unsigned                        nLength,
    bool                            bCsHold);

ADMW_RESULT admw_SpiReceive(
    ADMW_SPI_HANDLE hDevice,
    void *pTxData,
    void *pRxData,
    unsigned nLength,
    bool bCsHold);
#ifdef __cplusplus
}
#endif

/*!
 * @}
 */

#endif /* __ADMW_API_H__ */
