/* mbed Library - Sparkfun breakboard LCD_11062
 * This is using the Philips PCF8833 controller
 */

#include "LCD_11062.h"

#include "mbed.h"

using namespace mbed;

LCD_11062::LCD_11062(PinName mosi, PinName clk, PinName cs, PinName rst)
        : _spi(mosi, NC, clk)
        , _rst(rst)
        , _cs(cs) {
    _row       = 0;
    _column    = 0;
    _rows      = 16;
    _columns   = 16;
    _width     = 130;
    _height    = 130;
    _spifreq   = 80000000;
    charwidth  = 8;
    charheight = 8;
    foreground(0x000000);
    background(0xFFFF);
    reset();
}

void LCD_11062::reset() {
    long i;
    _cs  = 1;
    _rst = 0;
    _spi.format(9);
    _spi.frequency(_spifreq);
    wait(0.020);
    _rst = 1;
    wait(0.020);
    _select();
    command(0x11); //SLEEPOUT
    command(0x3A); //COLMOD
    data(0x05);    //0x03 = 16bits-per-pixel 5:6:5 mode
    command(0x36); //MADCTL
    //data(0x60);  // 0x60 = mirror x, vertical RAM write
    command(0x25); //SETCON
    data(0x30);    // contrast 0x30
    wait(0.002);
    command(0x29); //DISPON
    command(0x03); //BSTRON

    command(0x2B); //Row Address Set
    data(0);
    data(131);

    command(0x2A); //Column Address Set
    data(0);
    data(131);

    command(0x2C); //RAMWR
    //Clear RAM to black
    for (i = 0; i < (131 * 131); i++) {
        data(0);
        data(0);
    }
    _deselect();
}

void LCD_11062::command(int value) {
    _spi.write(value & 0xFF);
}

void LCD_11062::data(int value) {
    _spi.write(value | 0x100);
}

void LCD_11062::_select() {
    _spi.format(9);
    _spi.frequency(_spifreq);
    _cs = 0;
}

void LCD_11062::_deselect() {
    _cs = 1;
}

void LCD_11062::_window(int x, int y, int width, int height) {
    int x1, x2, y1, y2;
    x1 = 130 - (x+width) + 1;
    y1 = 130 - (y+height) + 1;
    x2 = x1 + width - 1;
    y2 = y1 + height - 1 ;
    command(0x2A); //CASET column
    data(x1);
    data(x2);
    command(0x2B); //PASET page
    data(y1);
    data(y2);
    command(0x2C); //RAMWR start write to ram
}

void LCD_11062::_putp(int colour) {
    int outcolor;
    outcolor = (colour & 0xFF)>>3;
    outcolor += (((colour>>8) & 0xFF)>>2)<<5;
    outcolor += (((colour>>16) & 0xFF)>>3)<<11;
    data((outcolor >> 8) & 0xFF);
    data(outcolor & 0xFF);
}

const unsigned char FONT8x8[97][8] = {
    0x08,0x08,0x08,0x00,0x00,0x00,0x00,0x00, // columns, rows, num_bytes_per_char
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00, // space 0x20
    0x30,0x78,0x78,0x30,0x30,0x00,0x30,0x00, // !
    0x6C,0x6C,0x6C,0x00,0x00,0x00,0x00,0x00, // "
    0x6C,0x6C,0xFE,0x6C,0xFE,0x6C,0x6C,0x00, // #
    0x18,0x3E,0x60,0x3C,0x06,0x7C,0x18,0x00, // $
    0x00,0x63,0x66,0x0C,0x18,0x33,0x63,0x00, // %
    0x1C,0x36,0x1C,0x3B,0x6E,0x66,0x3B,0x00, // &
    0x30,0x30,0x60,0x00,0x00,0x00,0x00,0x00, // '
    0x0C,0x18,0x30,0x30,0x30,0x18,0x0C,0x00, // (
    0x30,0x18,0x0C,0x0C,0x0C,0x18,0x30,0x00, // )
    0x00,0x66,0x3C,0xFF,0x3C,0x66,0x00,0x00, // *
    0x00,0x30,0x30,0xFC,0x30,0x30,0x00,0x00, // +
    0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x30, // ,
    0x00,0x00,0x00,0x7E,0x00,0x00,0x00,0x00, // -
    0x00,0x00,0x00,0x00,0x00,0x18,0x18,0x00, // .
    0x03,0x06,0x0C,0x18,0x30,0x60,0x40,0x00, // / (forward slash)
    0x3E,0x63,0x63,0x6B,0x63,0x63,0x3E,0x00, // 0 0x30
    0x18,0x38,0x58,0x18,0x18,0x18,0x7E,0x00, // 1
    0x3C,0x66,0x06,0x1C,0x30,0x66,0x7E,0x00, // 2
    0x3C,0x66,0x06,0x1C,0x06,0x66,0x3C,0x00, // 3
    0x0E,0x1E,0x36,0x66,0x7F,0x06,0x0F,0x00, // 4
    0x7E,0x60,0x7C,0x06,0x06,0x66,0x3C,0x00, // 5
    0x1C,0x30,0x60,0x7C,0x66,0x66,0x3C,0x00, // 6
    0x7E,0x66,0x06,0x0C,0x18,0x18,0x18,0x00, // 7
    0x3C,0x66,0x66,0x3C,0x66,0x66,0x3C,0x00, // 8
    0x3C,0x66,0x66,0x3E,0x06,0x0C,0x38,0x00, // 9
    0x00,0x18,0x18,0x00,0x00,0x18,0x18,0x00, // :
    0x00,0x18,0x18,0x00,0x00,0x18,0x18,0x30, // ;
    0x0C,0x18,0x30,0x60,0x30,0x18,0x0C,0x00, // <
    0x00,0x00,0x7E,0x00,0x00,0x7E,0x00,0x00, // =
    0x30,0x18,0x0C,0x06,0x0C,0x18,0x30,0x00, // >
    0x3C,0x66,0x06,0x0C,0x18,0x00,0x18,0x00, // ?
    0x3E,0x63,0x6F,0x69,0x6F,0x60,0x3E,0x00, // @ 0x40
    0x18,0x3C,0x66,0x66,0x7E,0x66,0x66,0x00, // A
    0x7E,0x33,0x33,0x3E,0x33,0x33,0x7E,0x00, // B
    0x1E,0x33,0x60,0x60,0x60,0x33,0x1E,0x00, // C
    0x7C,0x36,0x33,0x33,0x33,0x36,0x7C,0x00, // D
    0x7F,0x31,0x34,0x3C,0x34,0x31,0x7F,0x00, // E
    0x7F,0x31,0x34,0x3C,0x34,0x30,0x78,0x00, // F
    0x1E,0x33,0x60,0x60,0x67,0x33,0x1F,0x00, // G
    0x66,0x66,0x66,0x7E,0x66,0x66,0x66,0x00, // H
    0x3C,0x18,0x18,0x18,0x18,0x18,0x3C,0x00, // I
    0x0F,0x06,0x06,0x06,0x66,0x66,0x3C,0x00, // J
    0x73,0x33,0x36,0x3C,0x36,0x33,0x73,0x00, // K
    0x78,0x30,0x30,0x30,0x31,0x33,0x7F,0x00, // L
    0x63,0x77,0x7F,0x7F,0x6B,0x63,0x63,0x00, // M
    0x63,0x73,0x7B,0x6F,0x67,0x63,0x63,0x00, // N
    0x3E,0x63,0x63,0x63,0x63,0x63,0x3E,0x00, // O
    0x7E,0x33,0x33,0x3E,0x30,0x30,0x78,0x00, // P 0x50
    0x3C,0x66,0x66,0x66,0x6E,0x3C,0x0E,0x00, // Q
    0x7E,0x33,0x33,0x3E,0x36,0x33,0x73,0x00, // R
    0x3C,0x66,0x30,0x18,0x0C,0x66,0x3C,0x00, // S
    0x7E,0x5A,0x18,0x18,0x18,0x18,0x3C,0x00, // T
    0x66,0x66,0x66,0x66,0x66,0x66,0x7E,0x00, // U
    0x66,0x66,0x66,0x66,0x66,0x3C,0x18,0x00, // V
    0x63,0x63,0x63,0x6B,0x7F,0x77,0x63,0x00, // W
    0x63,0x63,0x36,0x1C,0x1C,0x36,0x63,0x00, // X
    0x66,0x66,0x66,0x3C,0x18,0x18,0x3C,0x00, // Y
    0x7F,0x63,0x46,0x0C,0x19,0x33,0x7F,0x00, // Z
    0x3C,0x30,0x30,0x30,0x30,0x30,0x3C,0x00, // [
    0x60,0x30,0x18,0x0C,0x06,0x03,0x01,0x00, // \ (back slash)
    0x3C,0x0C,0x0C,0x0C,0x0C,0x0C,0x3C,0x00, // ]
    0x08,0x1C,0x36,0x63,0x00,0x00,0x00,0x00, // ^
    0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFF, // _
    0x18,0x18,0x0C,0x00,0x00,0x00,0x00,0x00, // ` 0x60
    0x00,0x00,0x3C,0x06,0x3E,0x66,0x3B,0x00, // a
    0x70,0x30,0x3E,0x33,0x33,0x33,0x6E,0x00, // b
    0x00,0x00,0x3C,0x66,0x60,0x66,0x3C,0x00, // c
    0x0E,0x06,0x3E,0x66,0x66,0x66,0x3B,0x00, // d
    0x00,0x00,0x3C,0x66,0x7E,0x60,0x3C,0x00, // e
    0x1C,0x36,0x30,0x78,0x30,0x30,0x78,0x00, // f
    0x00,0x00,0x3B,0x66,0x66,0x3E,0x06,0x7C, // g
    0x70,0x30,0x36,0x3B,0x33,0x33,0x73,0x00, // h
    0x18,0x00,0x38,0x18,0x18,0x18,0x3C,0x00, // i
    0x06,0x00,0x06,0x06,0x06,0x66,0x66,0x3C, // j
    0x70,0x30,0x33,0x36,0x3C,0x36,0x73,0x00, // k
    0x38,0x18,0x18,0x18,0x18,0x18,0x3C,0x00, // l
    0x00,0x00,0x66,0x7F,0x7F,0x6B,0x63,0x00, // m
    0x00,0x00,0x7C,0x66,0x66,0x66,0x66,0x00, // n
    0x00,0x00,0x3C,0x66,0x66,0x66,0x3C,0x00, // o
    0x00,0x00,0x6E,0x33,0x33,0x3E,0x30,0x78, // p
    0x00,0x00,0x3B,0x66,0x66,0x3E,0x06,0x0F, // q
    0x00,0x00,0x6E,0x3B,0x33,0x30,0x78,0x00, // r
    0x00,0x00,0x3E,0x60,0x3C,0x06,0x7C,0x00, // s
    0x08,0x18,0x3E,0x18,0x18,0x1A,0x0C,0x00, // t
    0x00,0x00,0x66,0x66,0x66,0x66,0x3B,0x00, // u
    0x00,0x00,0x66,0x66,0x66,0x3C,0x18,0x00, // v
    0x00,0x00,0x63,0x6B,0x7F,0x7F,0x36,0x00, // w
    0x00,0x00,0x63,0x36,0x1C,0x36,0x63,0x00, // x
    0x00,0x00,0x66,0x66,0x66,0x3E,0x06,0x7C, // y
    0x00,0x00,0x7E,0x4C,0x18,0x32,0x7E,0x00, // z
    0x0E,0x18,0x18,0x70,0x18,0x18,0x0E,0x00, // {
    0x0C,0x0C,0x0C,0x00,0x0C,0x0C,0x0C,0x00, // |
    0x70,0x18,0x18,0x0E,0x18,0x18,0x70,0x00, // }
    0x3B,0x6E,0x00,0x00,0x00,0x00,0x00,0x00, // ~
    0x1C,0x36,0x36,0x1C,0x00,0x00,0x00,0x00  // DEL
};

void LCD_11062::locate(int column, int row) {
    _row = row;
    _column = column;
}

void LCD_11062::newline() {
    _column = 0;
    _row++;
    if (_row >= _rows) {
        _row = 0;
    }
}

int LCD_11062::_putc(int value) {
    int x = _column * charwidth;  // FIXME: Char sizes
    int y = _row * charheight;
    bitblit(x+1 , y+1 , charwidth, charheight, (char*)&(FONT8x8[value - 0x1F][0]));
    _column++;
    if (_column >= _columns) {
        _row++;
        _column = 0;
    }
    if (_row >= _rows) {
        _row = 0;
    }
    return value;
}

void LCD_11062::cls() {
    fill(0, 0, _width, _height, _background);
    _row = 0;
    _column = 0;
}

int LCD_11062::width() {
    return _width;
}

int LCD_11062::height() {
    return _height;
}

int LCD_11062::columns() {
    return _columns;
}

int LCD_11062::rows() {
    return _rows;
}

void LCD_11062::window(int x, int y, int width, int height) {
    _select();
    _window(x, y, width, height);
    _deselect();
}

void LCD_11062::putp(int colour) {
    _select();
    _putp(colour);
    _deselect();
}

void LCD_11062::pixel(int x, int y, int colour) {
    _select();
    _window(x, y, 1, 1);
    _putp(colour);
    _deselect();
}

void LCD_11062::fill(int x, int y, int width, int height, int colour) {
    _select();
    _window(x, y, width, height);
    for (int i=0; i<width*height; i++) {
        _putp(colour);
    }
    _window(0, 0, _width, _height);
    _deselect();
}

void LCD_11062::blit(int x, int y, int width, int height, const int* colour) {
    _select();
    _window(x, y, width, height);
    for (int i=0; i<width*height; i++) {
        _putp(colour[i]);
    }
    _window(0, 0, _width, _height);
    _deselect();
}

void LCD_11062::foreground(int v) {
    _foreground = v;
}

void LCD_11062::background(int v) {
    _background = v;
}

void LCD_11062::bitblit(int x, int y, int width, int height, const char* bitstream) {
    int bytediv = (width*height)>>3;
    _select();
    _window(x, y, width, height);
    for (int i=height*width-1; i>=0; i--) {
        unsigned char byte = i / bytediv;
        unsigned char bit = (i % width)/(width >> 3);
        int colour = (((bitstream[byte] << bit) & 0x80) ? _foreground : _background);
        _putp(colour);
    }
    _window(0, 0, _width, _height);
    _deselect();
}
