/* This code based on mbed TextLCD Library, for a 4-bit LCD based on HD44780,
 * Copyright (c) 2007-2010, sford, http://mbed.org
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

#include "mbed.h"
#include "include/menbedDisplayHD44780.h"
#include "menbedDisplay.h"

MenbedDisplayHD44780::MenbedDisplayHD44780 (PinName rs, PinName e, 
    PinName d4, PinName d5, PinName d6, PinName d7, LCDSize size) : 
    rs(rs), e(e), d(d4, d5, d6, d7), size(size) 
{
    this->e  = 1;
    this->rs = 0;
    
    upArrowVisible = false;
    downArrowVisible = false;
    topLineSelected = false;
    bottomLineSelected = false;
    cursorCol = -1;
    cursorRow = -1;

    wait(0.015);        // Wait 15ms to ensure powered up

    // send "Display Settings" 3 times (Only top nibble of 0x30 as we've got 4-bit bus)
    for (int i=0; i<3; i++) {
        writeByte(0x3);
        wait(0.00164);  // this command takes 1.64ms, so wait for it
    }
    
    writeByte(0x2);     // 4-bit mode
    wait(0.000040f);    // most instructions take 40us

    writeCommand(0x28); // Function set 001 BW N F - -
    writeCommand(0x0C); // Display on but keep cursor and blinking off
    writeCommand(0x6);  // Cursor Direction and Display Shift : 0000 01 CD S (CD 0-left, 1-right S(hift) 0-no, 1-yes
    
    loadCustomChars();
    
    clear();
}


bool MenbedDisplayHD44780::writeLine (const char *line, uint8_t row)
{
    int i = 0;
    int cursorPos = -1;

    if (row >= rows())
        return false;

    // Skip writing to the left-most column to leave space for the selector
    // and up/down arrows that will be filled in later.
    gotoPosition (row, 1);

    while ((line[i] != '\0') && (line[i] != '\n') && (i+1 < columns()))
    {
        // Place the cursor at the end of the active parameter, when it exists.
        // A parameter is active when it is highlighted as indicated by the 
        // MSB of the character code being set.
        if ((i > 0) && (line[i-1] & 0x80) && !(line[i] & 0x80))
            cursorPos = i-1;

        // Print each character to the display after clearing its MSB so that
        // it prints as a standard ASCII character.
        writeData (line[i] & ~0x80);
        i++;
    }
    
    // If the first character of the line is not highlighted but the last is,
    // a parameter must be selected for editing, and we place the cursor at
    // the end of the parameter which, in this special case, corresponds to
    // the end of the line.
    if (!(line[0] & 0x80) && (line[i-1] & 0x80))
        cursorPos = i-1;

    // Fill the remainder of the row with spaces to overwrite any old data
    while (i++ < columns() - 1)
        writeData(' ');

    // Now go back and fill in the selector and up/down arrows
    gotoPosition(row, 0);
    if ((line[0] & 0x80) && (cursorPos == -1))
    {
        if (row == 0)
            topLineSelected = true;
        else if (row == rows() - 1)
            bottomLineSelected = true;
    
        if ((row == 0) && upArrowVisible)
            writeData (CharUP_SELECT);
        else if ((row == rows() - 1) && downArrowVisible)
            writeData (CharDOWN_SELECT);
        else
            writeData (CharSELECT);
    }
    else
    {
        if (row == 0)
            topLineSelected = false;
        else if (row == rows() - 1)
            bottomLineSelected = false;
    
        if ((row == 0) && upArrowVisible)
            writeData (CharUP);
        else if ((row == rows() - 1) && downArrowVisible)
            writeData (CharDOWN);
        else
            writeData (' ');
    }

    // If a parameter is being edited, we turn on the blinking cursor.
    if (cursorPos != -1)
    {
        cursorRow = row;
        cursorCol = cursorPos + 1;
        gotoPosition (cursorRow, cursorCol); // Add 1 to account for selector col.
        cursorOn();
    }
    // If this line used to contain the cursor but should not any more reset
    // the cursor row and column position and turn it off.
    else if (row == cursorRow)
    {
        cursorRow = -1;
        cursorCol = -1;
        cursorOff();
    }

    return true;
}


void MenbedDisplayHD44780::showUpArrow (bool show)
{
    upArrowVisible = show;
    
    gotoPosition (0, 0);
    
    if (show && topLineSelected)
        writeData (CharUP_SELECT);
    else if (!show && topLineSelected)
        writeData (CharSELECT);
    else if (show && !topLineSelected)
        writeData (CharUP);
    else
        writeData(' ');
        
    // Return cursor to its original location
    if ((cursorRow >= 0) && (cursorCol >= 0))
        gotoPosition (cursorRow, cursorCol);
}


void MenbedDisplayHD44780::showDownArrow (bool show)
{
    downArrowVisible = show;
    
    gotoPosition (rows() - 1, 0);
    
    if (show && bottomLineSelected)
        writeData (CharDOWN_SELECT);
    else if (!show && bottomLineSelected)
        writeData (CharSELECT);
    else if (show && !bottomLineSelected)
        writeData (CharDOWN);
    else
        writeData(' ');
        
    // Return cursor to its original location
    if ((cursorRow >= 0) && (cursorCol >= 0))
        gotoPosition (cursorRow, cursorCol);
}


uint8_t MenbedDisplayHD44780::getLines (void)
{
    return rows();
}


uint8_t MenbedDisplayHD44780::getLineLength (void)
{
    return columns();
}


void MenbedDisplayHD44780::clear()
{
    writeCommand (0x01); // Clear, and set cursor to 0
    wait (0.050f);
    gotoPosition (0, 0);
}


bool MenbedDisplayHD44780::gotoPosition(int row, int column)
{
    if ((column < 0) || (column >= columns()) || (row < 0) || (row >= rows()))
        return false;

    writeCommand (address (row, column));
    return true;
}


void MenbedDisplayHD44780::cursorOn()
{
    writeCommand(0x0D); // Display and blinking on, cursor off
}


void MenbedDisplayHD44780::cursorOff()
{
    writeCommand(0x0C); // Display on, cursor and blinking off
}


void MenbedDisplayHD44780::writeCommand(int command) {
    rs = 0;
    writeByte(command);
}


void MenbedDisplayHD44780::writeData(int data) {
    rs = 1;
    writeByte(data);
}


void MenbedDisplayHD44780::writeByte(int value) {
    d = value >> 4;
    wait(0.000040f); // most instructions take 40us
    e = 0;
    wait(0.000040f);
    e = 1;
    d = value >> 0;
    wait(0.000040f);
    e = 0;
    wait(0.000040f);  // most instructions take 40us
    e = 1;
}


int MenbedDisplayHD44780::address(int row, int column) {
    switch (size) {
        case LCD20x4:
            switch (row) {
                case 0:
                    return 0x80 + column;
                case 1:
                    return 0xc0 + column;
                case 2:
                    return 0x94 + column;
                case 3:
                    return 0xd4 + column;
            }
        case LCD16x2B:
            return 0x80 + (row * 40) + column;
        case LCD16x2:
        case LCD20x2:
        default:
            return 0x80 + (row * 0x40) + column;
    }
}


int MenbedDisplayHD44780::columns() 
{
    switch (size) {
        case LCD20x4:
        case LCD20x2:
            return 20;
        case LCD16x2:
        case LCD16x2B:
        default:
            return 16;
    }
}


int MenbedDisplayHD44780::rows() 
{
    switch (size) {
        case LCD20x4:
            return 4;
        case LCD16x2:
        case LCD16x2B:
        case LCD20x2:
        default:
            return 2;
    }
}


void MenbedDisplayHD44780::loadCustomChars (void)
{

    // Up arrow
    writeCommand(0x40 + (8 * CharUP));
    writeData(0x04);
    writeData(0x0E);
    writeData(0x1F);
    writeData(0x00);
    writeData(0x00);
    writeData(0x00);
    writeData(0x00);
    writeData(0x00);
    
    // Up arrow with selector
    writeCommand(0x40 + (8 * CharUP_SELECT));
    writeData(0x04);
    writeData(0x0E);
    writeData(0x1F);
    writeData(0x00);
    writeData(0x1F);
    writeData(0x1F);
    writeData(0x00);
    writeData(0x00);
    
    // Selector alone
    writeCommand(0x40 + (8 * CharSELECT));
    writeData(0x00);
    writeData(0x00);
    writeData(0x00);
    writeData(0x1F);
    writeData(0x1F);
    writeData(0x00);
    writeData(0x00);
    writeData(0x00);    
    
    // Down arrow with selector
    writeCommand(0x40 + (8 * CharDOWN_SELECT));
    writeData(0x00);
    writeData(0x00);
    writeData(0x1F);
    writeData(0x1F);
    writeData(0x00);
    writeData(0x1F);
    writeData(0x0E);
    writeData(0x04);
    
    // Down arrow
    writeCommand(0x40 + (8 * CharDOWN));
    writeData(0x00);
    writeData(0x00);
    writeData(0x00);
    writeData(0x00);
    writeData(0x00);
    writeData(0x1F);
    writeData(0x0E);
    writeData(0x04);        
}