#include "mbed.h"
#include "ble/BLE.h"
#include "ble/services/HeartRateService.h"
#include "ble/services/BatteryService.h"
#include "ble/services/DeviceInformationService.h"
#include "SerialGPS.h"



Serial pc(USBTX, USBRX);

/**
 * PC_10 - TX pin (RX on the GPS module side)
 * PC_11 - RX pin (TX on the GPS module side)
 * 9600 - GPS baud rate
 */

SerialGPS gps(PC_10, PC_11, 9600);

DigitalOut led1(LED1);

BLE  ble;

const static char     DEVICE_NAME[]        = "NUCLEO_GPS";
static const uint16_t uuid16_list[]        = {GattService::UUID_HEART_RATE_SERVICE,
                                              GattService::UUID_DEVICE_INFORMATION_SERVICE};
static volatile bool  triggerSensorPolling = false;

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    ble.gap().startAdvertising(); // restart advertising
}

void periodicCallback(void)
{
    led1 = !led1; /* Do blinky on LED1 while we're waiting for BLE events */
    /* Note that the periodicCallback() executes in interrupt context, so it is safer to do
     * heavy-weight sensor polling from the main thread. */
    triggerSensorPolling = true;
}

int main(void)
{
    int k;
    led1 = 1;
    Ticker ticker;
    ticker.attach(periodicCallback, 1); // blink LED every second

    ble.init();
    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service. */
    uint16_t hrmCounter = 100; 
    HeartRateService hrService(ble, hrmCounter, HeartRateService::LOCATION_FINGER);

    /* Setup auxiliary service. */
    DeviceInformationService deviceInfo(ble, "ARM", "Model1", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");

    /* Setup advertising. */
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::GENERIC_HEART_RATE_SENSOR);
    ble.gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms */
    ble.gap().startAdvertising();

    pc.printf("Start loop\n\r");
    // infinite loop
    while (1) {
        // check for trigger from periodicCallback()
        if (triggerSensorPolling && ble.getGapState().connected){
            triggerSensorPolling = false;
            pc.printf("Waiting for sample..\n\r");
            if (gps.sample()) {
                for(k=0;k<=256;k++){
                    pc.printf("%c", gps.msg[k]);
                }
                    pc.printf("\n\r");
                    pc.printf("sats %d, lat %f, lont %f, alt %f, geoid %f, time %f\n\r", gps.sats, gps.latitude, gps.longitude, gps.alt, gps.geoid, gps.time);
                    hrService.updateHeartRate((uint16_t)(gps.longitude*1000));
            }
        } else {
            ble.waitForEvent(); // low power wait for event
        }
    }
}