/* mbed SCA61T Single Axis Inclinometer
* Copyright (c) 2010 Veikko Soininen
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#ifndef _SCA61T_H_
#define _SCA61T_H_

#include "mbed.h"

/** SCA61T Inclinometer class
*
* Example:
* @code
* // Send temperature and angle to serial port every one second.
* #include "mbed.h"
* #include "SCA61T.h"
*
* SCA61T sca61t(p11,p12,p13,p21,0);                  // MOSI, MISO, SCLK, CSB, device selection
*                                                    // 0=SCA61T-FAHH1G, 1=SCA61T-FA1H1G 
* Serial pc(USBTX, USBRX);
*
* int main(void)
* {
*    while(1)
*    {
*        pc.printf("%i [C]\r\n",sca61t.ReadTemp());      // Writes the temperature to serial port.
*        pc.printf("%.1f [deg]\r\n",sca61t.ReadX());     // Writes the angle to serial port.
*        wait(1);                                        // Waits for one second and repeats.
*    }
* }
* @endcode
*/

class SCA61T {
    public:
        SCA61T(PinName mosi, PinName miso, PinName sclk, PinName csb, int device_sel);       
        ~SCA61T() {};
        
        /** Reads the angle
        * 
        * @returns angle in degrees
        */
        float ReadX();
        
        /** Reads the temperature
        * 
        * @returns temperature in Celcius
        */
        int8_t ReadTemp();
        
        /** Sets the sensor to measurement mode
        * 
        * Default mode, not needed to call
        * first
        */
        void MeasMode();
        
        /** Reads the status register
        * 
        * @returns value of status register
        */        
        uint8_t ReadStatus();
        
        /** Reloads NV.
        *
        */
        void ReloadNV();
        
        /** Sets the sensor to self test mode
        * 
        * Discharges the sensor element. De-activated
        * by MeasMode().
        */
        void SelfTest();
                
    private:
        SPI SPI_m;
        DigitalOut CSB_m;
        
        void SPI_SendByte(uint8_t byte);
        uint8_t SPI_ReadReg(uint8_t reg);
        void SPI_ReadWord(uint8_t cmd, char* table);
        void SPI_Command(uint8_t cmd);
};

#endif
