#include "mbed.h"
#include "rtos.h"

#include "main.h"
#include "real_time_clock.h"
#include "temp_hum_sensor.h"
#include "motor_cnt.h"
#include "light_sensor.h"

#define TASKS_NUMBER    (sizeof(taskList)/sizeof(taskList[0]))

Mutex mutexData;
SensorData data;
Mutex mutexPCComm;
Serial pc(USBTX,USBRX);

Serial xbee(D10,D2);//(PB_6, PA_10);

volatile bool sensors_running = false;

static struct task_definition_struct taskList[] = {
    { temp_hum_thread,    &temp_hum_init, NULL, NULL, osPriorityNormal, 100, NULL, 0 },
    { light_thread,    &light_init, NULL, NULL, osPriorityNormal, 100, NULL, 0 }
};


int main()
{
    pc.baud(9600);
    mutexPCComm.lock();
    pc.printf("MAIN: init\n");
    mutexPCComm.unlock();
    xbee.putc('S');
    // Init RTC
    time_t epoch_time, elapsed_time;
    epoch_time = rtc.get_epoch();
    mutexPCComm.lock();
    pc.printf("MAIN: time %s\n", ctime(&epoch_time));
    mutexPCComm.unlock();
    // Reset all data status
    mutexData.lock();
    data.DHTError = -128;
    mutexData.unlock();
    // Init all tasks
    initTasks();
    while(true) {
        epoch_time = rtc.get_epoch();
        // Start all threads
        sensors_running = true;
        for (int i = 0; i < TASKS_NUMBER; i++) {
            if (taskList[i].retVal == 1) {
                taskList[i].thread->signal_set(0x1);
            }
        }
        publishSensorData();
        // Stop all remain tasks
        sensors_running = false;
        elapsed_time = rtc.get_epoch()-epoch_time;

        mutexPCComm.lock();
        pc.printf("MAIN: elapsed %ld\n",(long)elapsed_time);
        mutexPCComm.unlock();
        if(10-(long)elapsed_time>0) {
            Thread::wait((10-elapsed_time)*1000);
        } else
            Thread::wait(1000);
    }
}

int publishSensorData(void)
{
    Timer t;
    int error = -1;
    time_t ts  = rtc.get_epoch();
    t.reset();
    t.start();
    while(error != 0 && t.read()<MAX_TIME_TO_WAIT_NODE_COMM) {
        mutexData.lock();
        // Check if all data are valid
        error = data.DHTError;
        if( error == 0) {
            // Send all info
            mutexPCComm.lock();
            pc.printf("Temp\t%f\n",data.temperature);
            pc.printf("Hum\t%f\n",data.humidity);
            pc.printf("DewP\t%f\n",data.dewPoint);
            pc.printf("Light\t%f\n",data.light);
            pc.printf("%ld\tDHTError:\t%d\n",(long)ts, data.DHTError);
            pc.printf("%ld\tDHTError:\t%d\n",(long)ts, data.lightError);
            mutexPCComm.unlock();
            xbee.printf("{%ld:{DHT:{temp:%4.2f,hum:%4.2f,dew:%4.2f}},{LIGHT:{light:%f}})\n",(long)ts, data.temperature,data.humidity,data.dewPoint,data.light);
            set_motor_direction(1);
            set_motor_speed(data.light);
        }
        mutexData.unlock();
        if(error!=0) {
            Thread::wait(3000);
        }
        mutexPCComm.lock();
        pc.printf("MAIN: req data\t%f\n",t.read());
        mutexPCComm.unlock();
    } // while
    t.stop();
    // Send valid info only
    if(error!=0) {
        mutexData.lock();
        if( data.DHTError == 0) {
            mutexPCComm.lock();
            pc.printf("Temp:\t%f\n",data.temperature);
            pc.printf("Hum:\t%f\n",data.humidity);
            pc.printf("DewP:\t%f\n",data.dewPoint);
            pc.printf("%ld\tDHTError:\t%d\n",(long)ts, data.DHTError);
            xbee.printf("{%ld:{DHT:{temp:%4.2f,hum:%4.2f,dew:%4.2f}}\n",(long)ts, data.temperature,data.humidity,data.dewPoint);
            mutexPCComm.unlock();
        }
        if( data.lightError == 0) {
            mutexPCComm.lock();
            pc.printf("Light\t%f\n",data.light);
            pc.printf("%ld\tlightError:\t%d\n",(long)ts, data.lightError);
            xbee.printf("{%ld:{LIGHT:{light:%f}})\n",(long)ts, data.light);
            mutexPCComm.unlock();
            set_motor_direction(1);
            set_motor_speed(data.light);
        }
        if( data.lightError != 0 || data.lightError != 0) {
            mutexPCComm.lock();
            pc.printf("%ld\tDHTError:\t%d\n",(long)ts, data.DHTError);
            pc.printf("%ld\tlightError:\t%d\n",(long)ts, data.lightError);
            xbee.printf("{%ld:{DHT:{error:\t%d},{LIGHT:{error:%f}}}\n",(long)ts, data.DHTError, data.light);
            mutexPCComm.unlock();
        }
        mutexData.unlock();
    }
    // Reset all data status
    mutexData.lock();
    data.DHTError = -128;
    data.lightError = -128;
    mutexData.unlock();
    return error;
}



static uint32_t initTasks(void)
{
    uint32_t i;
    uint32_t retval = 1;

    mutexPCComm.lock();
    pc.printf("initTasks: init\n");
    mutexPCComm.unlock();
    for (i = 0; i < TASKS_NUMBER; i++) {
        mutexPCComm.lock();
        pc.printf("initTasks: %d\n",i);
        mutexPCComm.unlock();
        if (taskList[i].init != NULL) {
            if ((taskList[i].retVal = ((*taskList[i].init)(taskList[i].init_args))) == 0)
                retval = 0;
        } else {
            taskList[i].retVal = 1;
        }
    }

    for (i = 0; i < TASKS_NUMBER; i++) {
        mutexPCComm.lock();
        pc.printf("initTasks: start %d\n",i);
        mutexPCComm.unlock();
        Thread::wait(taskList[i].delay);

        mutexPCComm.lock();
        pc.printf("initTasks: su1 %d\n",retval);
        mutexPCComm.unlock();
        if (taskList[i].retVal == 1) {
            taskList[i].thread = new Thread(taskList[i].task, taskList[i].task_args, taskList[i].priority);
        }

        mutexPCComm.lock();
        pc.printf("initTasks: finish su2 %d\n",retval);
        mutexPCComm.unlock();
    }

    mutexPCComm.lock();
    pc.printf("initTasks: finish %d\n",retval);
    mutexPCComm.unlock();
    return retval;
}