
 
#include "I2Cdev.h"
#include "MPU6050_6Axis_MotionApps20.h"
 
MPU6050 mpu;
 
const float M_PI = 3.14159265;
 
 
// MPU control/status vars
bool dmpReady = false;  // DMPの初期化が成功した場合はtrueに設定
uint8_t mpuIntStatus;   // センサの割り込みステータスを保持
uint8_t devStatus;      // デバイス動作後の状態 (0 = success, !0 = error)
uint16_t packetSize;    // expected DMP packet size (default is 42 bytes)
uint16_t fifoCount;     // count of all bytes currently in FIFO
uint8_t fifoBuffer[64]; // FIFO storage buffer
 
// orientation/motion vars
Quaternion q;           // [w, x, y, z]         quaternion container
VectorInt16 aa;         // [x, y, z]            accel sensor measurements
VectorInt16 aaReal;     // [x, y, z]            gravity-free accel sensor measurements
VectorInt16 aaWorld;    // [x, y, z]            world-frame accel sensor measurements
VectorFloat gravity;    // [x, y, z]            gravity vector
float euler[3];         // [psi, theta, phi]    Euler angle container
float ypr[3];           // [yaw, pitch, roll]   yaw/pitch/roll container and gravity vector
 
 
DigitalOut led1(LED1);
InterruptIn checkpin(p25);
Serial pc(USBTX, USBRX);
 
// ================================================================
// ===                     割り込み検出ルーチン                    ===
// ================================================================
 
volatile bool mpuInterrupt = false;     // センサの割り込みピンがHighになったかどうかを示します
void dmpDataReady() {
    mpuInterrupt = true;
}
 
 
// ================================================================
// ===                        メインルーチン                      ===
// ================================================================
void setup();
void loop();
 
int main() {
    setup();
    while(1) {
        loop();
    }
}
 
// ================================================================
// ===                      INITIAL SETUP                       ===
// ================================================================
 
void setup() {
 
    // initialize device
    pc.printf("Initializing I2C devices...\r\n");
    mpu.initialize();
 
    // verify connection
    pc.printf("Testing device connections...\r\n");
    if (mpu.testConnection()) pc.printf("MPU6050 connection successful\r\n");
    else pc.printf("MPU6050 connection failed\r\n");
    
    // wait for ready
    //Serial.println(F("\nSend any character to begin DMP programming and demo: "));
    //while (Serial.available() && Serial.read()); // empty buffer
    //while (!Serial.available());                 // wait for data
    //while (Serial.available() && Serial.read()); // empty buffer again
 
    // load and configure the DMP
    pc.printf("Initializing DMP...\r\n");
    devStatus = mpu.dmpInitialize();
    
    // make sure it worked (returns 0 if so)
    if (devStatus == 0) {
        // turn on the DMP, now that it's ready
        pc.printf("Enabling DMP...\r\n");
        mpu.setDMPEnabled(true);
 
        // enable Arduino interrupt detection
        pc.printf("Enabling interrupt detection (Arduino external interrupt 0)...\r\n");
        checkpin.rise(&dmpDataReady);
        
        mpuIntStatus = mpu.getIntStatus();
 
        // set our DMP Ready flag so the main loop() function knows it's okay to use it
        pc.printf("DMP ready! Waiting for first interrupt...\r\n");
        dmpReady = true;
 
        // get expected DMP packet size for later comparison
        packetSize = mpu.dmpGetFIFOPacketSize();
    } else {
        // ERROR!
        // 1 = initial memory load failed
        // 2 = DMP configuration updates failed
        // (if it's going to break, usually the code will be 1)
        
        pc.printf("DDMP Initialization failed (code ");
        pc.printf("%d", devStatus);
        pc.printf(")\r\n");
    }
 
}
 
 
 
// ================================================================
// ===                    MAIN PROGRAM LOOP                     ===
// ================================================================
 
void loop() {
    // DMPの初期化に失敗した場合、何もしない
    if (!dmpReady) return;
 
    // センサー割り込み or fifoオーバーフロー待ち
    while (!mpuInterrupt && fifoCount < packetSize) {
    }
 
    // reset interrupt flag and get INT_STATUS byte
    mpuInterrupt = false;
    mpuIntStatus = mpu.getIntStatus();
 
    // get current FIFO count
    fifoCount = mpu.getFIFOCount();
 
    // オーバーフローをチェック (我々のコードがあまりにも非効率的でない限り、これが起こることはありません)
    if ((mpuIntStatus & 0x10) || fifoCount == 1024) {
        // reset so we can continue cleanly
        mpu.resetFIFO();
        //Serial.println(F("FIFO overflow!"));
 
    // オーバーフローが無ければ、DMPデータ・レディ割り込みをチェック
    } else if (mpuIntStatus & 0x02) {
        // wait for correct available data length, should be a VERY short wait
        while (fifoCount < packetSize) fifoCount = mpu.getFIFOCount();
 
        // read a packet from FIFO
        mpu.getFIFOBytes(fifoBuffer, packetSize);
        
        // track FIFO count here in case there is > 1 packet available
        // (this lets us immediately read more without waiting for an interrupt)
        fifoCount -= packetSize;

            // display Euler angles in degrees
            mpu.dmpGetQuaternion(&q, fifoBuffer);
            mpu.dmpGetGravity(&gravity, &q);
            mpu.dmpGetYawPitchRoll(ypr, &q, &gravity);
//            printf("ypr\t");
            printf("%f,", ypr[0] * 180/M_PI);
            printf("%f,", ypr[1] * 180/M_PI);
            printf("%f,\n\r", ypr[2] * 180/M_PI);
            wait(0.1);
 
    }
}