#include "ADP5585.h"

extern Serial pc;

ADP5585::ADP5585(PinName sda, PinName scl, char adr) : i2c (sda, scl), _slaveAddress(adr << 1)
{
    i2c.frequency(ADP5585_FREQUENCY);
		i2c.start();
		i2c.stop();
	
    if (i2c.read(_slaveAddress, &data[0], 1) != 0) {
    	Check = ADP5585_OK;//"I2C_Keypad_Fail\r\n";
    	_check = false;
    }
    else {
        Check = ADP5585_FAIL;//"I2C_Keypad_Ok\r\n";
        _check = true;
    }   
    data[0] = 0;
}

ADP5585::~ADP5585(void)
{
    
}

static const uint8_t gpio_dir_reg[] = {
    ADP5585_ADR_GPIO_DIRECTION_A,
    ADP5585_ADR_GPIO_DIRECTION_B,
};

static const uint8_t gpo_data_reg[] = {
    ADP5585_ADR_GPO_DATA_OUT_A,
    ADP5585_ADR_GPO_DATA_OUT_B,
};


void ADP5585::setup(uint8_t row, uint8_t col, int (*sf)(int evt)){       
	//set row pin config
	for(uint8_t r=0;r<row;r++)
		activateRow(r);
	//set column pin config
	for(uint8_t c=0;c<col;c++)
		activateColumn(c);
	//enable the internal oscillator, INT pin deasserts for 50 ms and reasserts if an interrupt is pending.	
	writeRegister(ADP5585_ADR_GENERAL_CFG, ADP5585_GENERAL_CFG_OSC_EN | ADP5585_GENERAL_CFG_INT_CFG);
	
	for(int r=0;r<row;r++)
	{
	    for(int c=0;c<col;c++)
		{
			registerCallback(r, c, (*sf), ADP5585_RISING);
		}
	}
}

char ADP5585::getkey(int evt8_t){
    switch(evt8_t)
	{
		case ADP5585_KEY_1: return '1';
		case ADP5585_KEY_2: return '2';
		case ADP5585_KEY_3: return '3';
		case ADP5585_KEY_4: return '4';
		case ADP5585_KEY_5: return '5';
		case ADP5585_KEY_6: return '6';
		case ADP5585_KEY_7: return '7';
		case ADP5585_KEY_8: return '8';
		case ADP5585_KEY_9: return '9';
		case ADP5585_KEY_0: return '0';
		case ADP5585_KEY_M: return 'M';
		case ADP5585_KEY_B: return 'B';
		case ADP5585_KEY_U: return 'U';
		case ADP5585_KEY_D: return 'D';
		case ADP5585_KEY_Y: return 'Y';
		case ADP5585_KEY_N:	return 'N';
		default: return NULL;
	}
}

void ADP5585::createEventMatrix(void){
    //read the active rows and columns
    uint8_t activeRows = readRegister(ADP5585_ADR_PIN_CONFIG_A);
    uint8_t activeColumns = readRegister(ADP5585_ADR_PIN_CONFIG_B);
    
    //free all active events
    for(int i=0; i<numEvents; i++){
        free(eventMatrix[i]);
    }
    numEvents = 0;
    
    for(int i=0; i<7; i++){
        //if this row is active, set the event active on all active columns
        if((activeRows >> i) & 1){
            for(int j=0; j<7; j++){
                if((activeColumns >> j) & 1){
                    event *evt = (event *)malloc(sizeof(struct event));
                    evt->number = getEvent(i, j);
                    evt->status = false;
                    eventMatrix[numEvents] = evt;
                    numEvents++;
                }
            }
        }
    }
}

void ADP5585::update(void){
    int newEvents = readRegister(ADP5585_ADR_STATUS) & 0x1F;
    
    //update all events from the last read
    for(int i=0; i<numEvents; i++){
        event *evt = eventMatrix[i];
        if(evt->status == ADP5585_RISING)
            evt->status = ADP5585_HIGH;
        else if(evt->status == ADP5585_FALLING)
            evt->status = ADP5585_LOW;
    }
    
    //read all events in the fifo
    for(int i=0; i<newEvents; i++)
    {
    	uint8_t evt = readRegister(ADP5585_ADR_FIFO1);
        uint8_t evtNumber = evt & 0x7F;
        bool evtStatus = evt >> 7;
        
        //update the event matrix
        for(int j=0; j<numEvents; j++){
            event *e = eventMatrix[j];
            if(e->number == evtNumber){
                if(evtStatus)
                    e->status = ADP5585_RISING;
                else
                    e->status = ADP5585_FALLING;
            }
        }
    }
    
    //call any callbacks
    for(int i=0; i<numCallbacks; i++){
        s_callback *cb = callbacks[i];
        event *evt = eventMatrix[cb->evtIndex];
        if(cb->callbackType == evt->status)
            cb->f_callback(evt->number);
    }
}

bool ADP5585::registerCallback(uint8_t row, uint8_t col, int (*fn)(int evt), uint8_t type){
    uint8_t evtNumber = getEvent(row, col);
    for(int j=0; j<numEvents; j++){
        event *e = eventMatrix[j];
        if(e->number == evtNumber){
            s_callback *cb = (s_callback *)malloc(sizeof(struct s_callback));
            cb->callbackType = type;
            cb->evtIndex = j;
            cb->f_callback = fn;
            callbacks[numCallbacks] = cb;
            numCallbacks++;
            break;
        }
    }
}

void ADP5585::activateRow(uint8_t row){
    if(row <= 7){
        writeRegister(ADP5585_ADR_PIN_CONFIG_A, (1 << row) | readRegister(ADP5585_ADR_PIN_CONFIG_A));
        createEventMatrix();
    }
}

void ADP5585::activateColumn(uint8_t col){
    if(col <= 7){
        writeRegister(ADP5585_ADR_PIN_CONFIG_B, (1 << col) | readRegister(ADP5585_ADR_PIN_CONFIG_B));
        createEventMatrix();
    }
}

void ADP5585::gpioSetDirection(uint8_t gpio, uint8_t dir){
    if(gpio > 0 && gpio <= 19 && dir >= 0 && dir <= 1){
        uint8_t bit = ((gpio - 1) % 8);
        uint8_t reg = gpio_dir_reg[(gpio - 1) / 8];
        
        uint8_t toWrite = readRegister(reg);
        toWrite ^= (-dir ^ toWrite) & (1 << bit);
        
        writeRegister(reg, toWrite);
    }
}

bool ADP5585::writeRegister(uint8_t reg, uint8_t val){
	//Write the specified value to the specified register
	w[0] = reg;	//set register pointer
    w[1] = val;	//set value
    if (i2c.write(_slaveAddress, w, 2) != 0) return 0;
    
    return true;
}

void ADP5585::gpioWrite(uint8_t gpio, uint8_t val){
    //Write the specified value to the specified GPIO pin
    if(gpio > 0 && gpio <= 19 && val >= 0 && val <= 1){
        uint8_t bit = ((gpio - 1) % 8);
        uint8_t reg = gpo_data_reg[(gpio - 1) / 8];
        
        uint8_t toWrite = readRegister(reg);
        toWrite ^= (-val ^ toWrite) & (1 << bit);
        
        writeRegister(reg, toWrite);
    }
}

uint8_t ADP5585::readRegister(uint8_t reg){
		//Read the specified register
		char start = reg;	//set register pointer
    if (i2c.write(_slaveAddress, &start, 1, true) != 0) return 0;
    if (i2c.read(_slaveAddress, r, 1) != 0) return 0;
    
    return r[0];
}

uint8_t ADP5585::getEvent(uint8_t row, uint8_t col){
    return 1 + (row * ADP5585_NUM_COLUMNS) + col;
}
