#include "sensorinterface.h"
 
sensorinterface::sensorinterface() : i2c(p28,p27), readyLine(p29)
{ 
    readyLine.fall(this, &sensorinterface::readySet);
    readyLine.rise(this, &sensorinterface::readyUnset);
    
    this->ready = false;
    
    char buffer[255];
    
    #ifdef DEBUG
        printf("[SIF] Scanning for devices...\r\n");
    #endif
    
    for (char i=1; i<=254; i=i+2)
    {
        //last bit determines read/write not actual address
        if (this->i2c.read(i, &buffer[0], 1) ==0)
        {
            //returns 0 if ack i.e. a device is found
            sensor newSensor;
            newSensor = findType(newSensor, i);
            this->sensors[i] = newSensor;
            #ifdef DEBUG
                printf("[SIF] Device found at 0x%.2X, device count: %i\r\n", i, this->sensors.size());
            #endif
        }
    }
}

bool sensorinterface::isDataWaiting()
{
    return this->ready;
}

void sensorinterface::requestData()
{
    if (sensors.size() > 0)
    {
        currentSensor = sensors.begin();
        
        #ifdef DEBUG
            printf("[SIF] Beginning device query chain from 0x%.2X\r\n", currentSensor->first);
        #endif
        
        sendRequest(currentSensor->first);
    }
    else
    {
        #ifdef DEBUG
            printf("[SIF] Data requested, but no devices present\r\n");
        #endif
    }
}

void sensorinterface::readySet()
{
    this->ready = true;
}

void sensorinterface::readyUnset()
{
    this->ready = false;
}

#ifdef DEBUG
void sensorinterface::error(int e_num)
{
    printf("[SIF] Error %i\r",e_num);
}
#endif
 
int sensorinterface::update()
{
    char buffer[255];
    
    for (char i=1; i<=254; i=i+2)
    {
        //last bit determines read/write not actual address
        
        bool connected = (this->i2c.read(i, &buffer[0], 1) == 0);
        bool registered = (this->sensors.count(i) > 0);
        
        if (connected)
        {
            if (!registered)
            {
                //device added
                #ifdef DEBUG
                    printf("[SIF] Device added at %i\r\n", i);
                #endif
                sensor newSensor;
                newSensor = findType(newSensor, i);
                this->sensors[i] = newSensor;
            }
        }
        else if (registered)
        {
            if (!connected)
            {
                //device removed
                #ifdef DEBUG
                    printf("[SIF] Device removed at %i\r\n", i);
                #endif
                this->sensors.erase(i);
            }
        }
    }
    
    return 1;
}
 
 
sensor& sensorinterface::findType(sensor& s, int address)
{
    char response[I2C_TYPE_PACKET_SIZE];
    char cmd[1] = {I2C_TYPE_HEADER};
 
    this->i2c.write(address, cmd, 1);
    int type_timeout = 10;
    i2c.stop();
    i2c.start();
    while( (this->i2c.read(address, response, I2C_TYPE_PACKET_SIZE)) && (type_timeout) )
    {
        --type_timeout;
        if (type_timeout == 2)
        {
            #ifdef DEBUG
                error(3);
            #endif
            return s;
        }
    }

    if (response[0] != I2C_TYPE_HEADER)
    {
        #ifdef DEBUG
            error(1);
        #endif
    }

    s.type = response[1];
    s.packetSize = response[2];
    s.readingSize = response[3]; 
    return s;
}
 
int sensorinterface::sendRequest(char address)
{
    char cmd[1] = {I2C_REQUEST_HEADER};
 
    this->i2c.write(address, cmd, 1);
    
    return 1;
}
 
d_reply sensorinterface::readData()
{
    #ifdef DEBUG
        printf("[SIF] Reading data from current device (0x%.2X)\r\n", currentSensor->first);
    #endif
    char address = currentSensor->first;
    char cmd[1] = {I2C_DATA_HEADER};
    char bufSize = currentSensor->second.packetSize + 2;
    char buffer[bufSize];
    this->i2c.write(address, cmd, 1);
    wait(0.01);
    this->i2c.read(address, buffer, bufSize);
    
    d_reply reply;
    
    reply.type = currentSensor->second.type;
    
    for (int i = 2; i < bufSize; i += currentSensor->second.readingSize)
    {
        int r = 0;
        for (int j = 0; j < currentSensor->second.readingSize; j++)
        {
            r = r << 8;
            r += buffer[i+j];
        }
        reply.readings.push_back(r);
    }
 
    if (buffer[0] != I2C_DATA_HEADER)
    {
        //if it doesn't send the correct header bit, something has gone wrong
        #ifdef DEBUG
            error(0);
        #endif
    }
    
    if (currentSensor != sensors.end() && sensors.size() > 1 )
    {
        #ifdef DEBUG
            printf("[SIF] Continuing chain of devices, just read from device: 0x%.2X\r\n", currentSensor->first);
        #endif
        currentSensor++;
        sendRequest(currentSensor->first);
    }
    #ifdef DEBUG
    else
    {
        printf("[SIF] All devices have been read from, end of chain\r\n");
    }
    #endif
    
    return reply;
}