// Parent class for all addressable LED strips.
// Partially based on work by and (c) 2011 Jelmer Tiete
// whose library is ported from the Arduino implementation of Adafruit Industries
// found at: http://github.com/adafruit/LPD8806
// and their strips: http://www.adafruit.com/products/306
// Released under the MIT License: http://mbed.org/license/mit

// This is a pure virtual parent class for all LED strips, so that different types
// of strip may be used in a single array or container.

#include "mbed.h"

#ifndef LEDSTRIP_H
#define LEDSTRIP_H

/** Generic LED Strip
Pure virtual parent class for all types of LED strip
*/
class LedStrip
{
public:
    /** Create an LED strip
    @param pixelCount Number of RGB LEDs on the strip
    */
    LedStrip(uint16_t pixelCount);
    ~LedStrip();

    //! Initialise the LED strip
    virtual void begin(void)=0;
    //! Apply the new LED strip values
    virtual void show(void)=0;
    //! Blank the LED strip
    virtual void blank(void)=0;

    /** Pack RGB Color data
    @param red Amount of Red
    @param green Amount of Green
    @param blue Amount of Blue
    @returns Packed RGB color data for one pixel
    */
    static uint32_t Color(uint8_t red, uint8_t green, uint8_t blue);

    //! Number of RGB pixels
    uint16_t numPixels() {
        return numLEDs;
    }
    //! Number of bytes used for pixel colour data
    uint16_t numPixelBytes() {
        return numLEDs * 3;
    }
    /** Total brightness of all diodes\n
    * Use to check power budget
    @returns Sum total of all diodes (red + green + blue)
    */
    uint32_t total_luminance();

    /** Set Blue level of pixel
    @param pixNum Pixel Number
    @param blue Amount of Blue
    */
    void setPixelB(uint16_t pixNum, uint8_t blue);
    /** Set Green level of pixel
    @param pixNum Pixel Number
    @param green Amount of Green
    */
    void setPixelG(uint16_t pixNum, uint8_t green);
    /** Set Red level of pixel
    @param pixNum Pixel Number
    @param red Amount of Red
    */
    void setPixelR(uint16_t pixNum, uint8_t red);

    /** Set color of pixel
    @param pixNum Pixel Number
    @param color Packed RGB color data
    */
    void setPixelColor(uint16_t pixNum, uint32_t color);
    /** Set color of pixel
    @param pixNum Pixel Number
    @param red Amount of Red
    @param green Amount of Green
    @param blue Amount of Blue
    */
    void setPixelColor(uint16_t pixNum, uint8_t red, uint8_t green, uint8_t blue);
    /** Set color of all pixels
    @param *buffer Packed pixel data
    @param count Number of pixels
    */
    void setPackedPixels(uint8_t * buffer, uint32_t count);

protected:
    uint8_t *pixels;     // Holds LED color values
    uint16_t numLEDs;     // Number of RGB LEDs in strand
};
#endif
