/* This library is based on the Ser25lcxxx library by Hendrik Lipka
* It was adapted to flash memory chips on 19.2.2011 by Klaus Steinhammer
*
* It was further adapted to M25P* SPI Flash memory chips that do not
* support the "RDID" device ID instructions on 25.03.2014 by Richard Thompson
*
* The BSD license also applies to this code - have fun.
*/

/*
* Ser25lcxxx library
* Copyright (c) 2010 Hendrik Lipka
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#pragma once

// RTOS DMA-enabled version
#include "rtos.h"
#include "RTOS_SPI.h"

#include "mbed.h"

/**
An RTOS interface class to read and write M25P* serial SPI flash devices.
Supports M25P10-A and M25P40 ICs with no Device ID.
*/
class FlashM25PSpi
{
public:
    /**
        Create the flash interface class.
        It will autodetect the Signature of your device. If your device is not recognized, add the devices parameters to the device data structure.
        @param spi the RTOS_SPI port where the flash is connected. Must be set to speed matching your device.
        @param enable the pin name for the port where /CS is connected
        @param bits SPI bits format. If zero, leave SPI Format unchanged
        @param mode SPI mode.
    */
    FlashM25PSpi(RTOS_SPI *spi, PinName enable, int bits = 8, int mode = 3);

    /**
        Destroy the interface and power down the flash chip
    */
    ~FlashM25PSpi();

    /**
        Read a part of the flash memory into destination.
        The buffer must be pre-allocated by the caller.
        @param startAddr Flash address to start reading from. Doesn't need to match a page boundary
        @param destination Destination buffer
        @param len Number of bytes to read (must not exceed the end of memory)
        @return true if succeeded
    */
    bool read(uint32_t startAddr, void* destination, size_t len);

    /**
        Write the provided buffer into the flash memory.
        This function handles dividing the write into pages until the physical write has finished.
        Blocks until write completes.
        @note The flash must be in the erased (0xFF) state before starting a write cycle. This is not checked.
        @param startAddr Address to start writing. Doesn't need to match a page boundary
        @param data Source data buffer
        @param len the number of bytes to write (must not exceed the end of flash memory)
        @return false if the addresses are out of range
    */
    bool write(uint32_t startAddr, const void* data, size_t len);

    /**
        Erases the sector containing the given address to 0xFF.
        This erases several pages, refer to the IC datasheet or
        sectorSize() and pageSize() methods for memory organisation.
        @param addr Any address within the sector to be cleared
    */
    void eraseSector(uint32_t addr);

    //! Erases the whole flash to 0xFF
    void eraseMem();

    //! Flash size
    size_t flashSize() const {
        return _size;
    }

    //! Flash sector size
    size_t sectorSize() const {
        return _sectorSize;
    }

    //! Flash page size
    size_t pageSize() const {
        return _pageSize;
    }

private:
    bool writePage(uint32_t startAddr, const uint8_t* data, size_t len);
    int readStatus();
    void waitForWrite();
    void enableWrite();

    RTOS_SPI* _spi;

    DigitalOut _enable;
    size_t _size, _pageSize, _sectorSize;
};
