#pragma once

#include "USBDevice.h"

#include <stdint.h>

// See https://github.com/pbatard/libwdi/wiki/WCID-Devices

struct PACKED WindowsOsString
{
    uint8_t bLength; // Length of this structure, 0x12.
    uint8_t bDescriptorType; // String descriptor, 0x03 (STRING_DESCRIPTOR)
    uint8_t qwSignature[14]; // 'MSFT100' but in UTF-16 ( "M\0S\0F\0T\01\00\00\0" )
    uint8_t bMS_VendorCode; // Vendor code to retrieve OS feature desciptors. Can be changed. Example 0x01.
    uint8_t bPad; // Padding, 0x00.
};

// These headers are the same except for the reserved bytes, argh!
// Also the count is a word rather than a byte for extended properties.
struct PACKED CompatIdHeader
{
    uint32_t dwLength; // Length of the complete feature (CompatIdHeader + N * CompatIdFunction).
    uint16_t bcdVersion; // Descriptor's version number in BCD format. 0x0100 for MS OS descriptors 1.0
    uint16_t wIndex; // An index that identifies the OS feature descriptor.  (4 for CompatID, 5 for Extended Properties, etc.)
    uint8_t bCount; // Number of custom property sections.
    uint8_t reserved[7];
};

struct PACKED ExtendedPropertiesHeader
{
    uint32_t dwLength; // Length of the complete feature (FeatureHeader + N * ExtendedProperty).
    uint16_t bcdVersion; // Descriptor's version number in BCD format. 0x0100 for MS OS descriptors 1.0
    uint16_t wIndex; // An index that identifies the OS feature descriptor.  (4 for CompatID, 5 for Extended Properties, etc.)
    uint16_t wCount; // Number of custom property sections.
};

// For the CompatID feature the FeatureHeader is followed by
// 1 CompatIdFunction for a non-composite device.
// To be a WinUSB device the compatibleID filed must be "WINUSB" (in ASCII).
// I guess the subCompatibleID can be 0.
struct PACKED CompatIdFunction
{
    uint8_t bFirstInterfaceNumber;
    uint8_t reserved1;
    uint8_t compatibleID[8];
    uint8_t subCompatibleID[8];
    uint8_t reserved2[6];
};

// For a WinUSB device we also need an Extended Properties OS feature
// descriptor to register its GUID so we can find it in our app.
struct PACKED ExtendedPropertyGUID
{
    uint32_t dwSize; // The size of this custom property section.
    uint32_t dwPropertyDataType; // Property format.
    uint16_t wPropertyNameLength; // Property length.
    uint8_t bPropertyName[42]; // The property name (size equal to wPropertyNameLength). 42 if using DeviceInterfaceGUIDs, 40 otherwise.
    uint32_t dwPropertyDataLength;
    uint8_t bPropertyData[80]; // Property data (size equal to dwPropertyDataLength). 78 if using DeviceInterfaceGUID, 80 if using DeviceInterfaceGUIDs
};

// We can actually use Extended Properties to set the device icon and label
// (although WinUSB devices use the actual USB product name anyway).
//
// To register the WinUSB device GUID we use an extended property with the property
// name "DeviceInterfaceGUID" (as UTF-16) and the property name length to 40 bytes
// (19 chars + null) * 2 = 40 bytes.
// The bPropertyData is the UTF-16 string of the GUID, including curly braces, and the length is 78
// bytes (including the null terminator).
// E.g. the property data could be
// {8FE6D4D7-49DD-41E7-9486-49AFC6BFE475}
// Which is 38 characters, plus a null character. Each character is two bytes, so
// (38+1)*2 = 78.

struct PACKED CompatIdData
{
    CompatIdHeader header;
    CompatIdFunction data[1]; // We only want one compat ID - to set the compatibleID to WINUSB.
};

struct PACKED ExtendedPropertyData
{
    ExtendedPropertiesHeader header;
    ExtendedPropertyGUID data[1]; // Only one extended property for now - to set the GUID.
};

class WinUSBDevice : public USBDevice
{
public:
    // guid should be of the format "{a451588c-7230-4076-8456-9e544165d90c}"
    WinUSBDevice(uint16_t vendor_id, uint16_t product_id, uint16_t product_release, const char* guid);

    // Handle vendor setup requests. Return true if we handle the request, otherwise false to let mBed do its
    // default action, which is to call requestSetup().
    // This is called in an ISR context.
    bool USBCallback_request();// override;
private:
    WindowsOsString osStringDesc;
    CompatIdData compatIdData;
    ExtendedPropertyData extendedPropertyData;
};

