/* mbed library for 240*320 pixel display TFT based on HX8347D LCD Controller
 * Copyright (c) 2011 Peter Drescher - DC2PD
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


// fix bmp padding for Bitmap function
// speed up pixel
// 30.12.11 fix cls
// 11.03.12 use DMA to speed up
// 15.03.12 use SSEL for TFT CS to enable DMA Register writes
// 06.04.12 fix SSEL CS problem
// 06.04.12 use direct access to the spi register to speed up the library.
// 11.09.12 switch back to using io pin as cs to avoid problems with SSEL CS.
// 21.09.12 fix Bug in BMP_16 
// 11.10.12 patch from Hans Bergles to get SPI1 working again
// 03.02.13 add a switch to switch off DMA use for LPC11U24

#include "SPI_TFTx2.h"
#include "mbed.h"

#define BPP         16                  // Bits per pixel                

#if defined TARGET_LPC1768 
#define USE_DMA
#endif

#if defined NO_DMA
#undef USE_DMA
#endif

//extern Serial pc;
//extern DigitalOut xx;     // debug !!

SPI_TFTx2::SPI_TFTx2(PinName mosi, PinName miso, PinName sclk, PinName cs0, PinName cs1, PinName reset, const char *name)
    : _spi(mosi, miso, sclk), _cs0(cs0), _cs1(cs1), _reset(reset), GraphicsDisplay(name)
{
    orientation = 0;
    char_x = 0;
    _cs0 = 1;
    _cs1 = 1;

    if (mosi == p11 || mosi == P0_18) spi_port = 0;  // we must know the used SPI port to setup the DMA
    else spi_port = 1;
    seldisp = 2;
    tft_reset();
}

int SPI_TFTx2::width()
{
    if (orientation == 0 || orientation == 2) return 240;
    else return 320;
}


int SPI_TFTx2::height()
{
    if (orientation == 0 || orientation == 2) return 320;
    else return 240;
}


void SPI_TFTx2::set_orientation(unsigned int o)
{
    orientation = o;
    switch (orientation) {
        case 0:
            wr_reg(0x16, 0x08);
            break;
        case 1:
            wr_reg(0x16, 0x68);
            break;
        case 2:
            wr_reg(0x16, 0xC8);
            break;
        case 3:
            wr_reg(0x16, 0xA8);
            break;
    }
    WindowMax();
}

void SPI_TFTx2::set_display(unsigned int o)
{
    seldisp = o;
}

// write command to tft register

void SPI_TFTx2::wr_cmd(unsigned char cmd)
{
    unsigned short spi_d;
    spi_d =  0x7000 | cmd ;

    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    
    if (spi_port == 0) {    // TFT on SSP0
        LPC_SSP0->DR = spi_d;
        // we have to wait for SPI IDLE to set CS back to high
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI0 not idle
    } else {
        LPC_SSP1->DR = spi_d;
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    }
    _cs0 = 1;
    _cs1 = 1;
}



void SPI_TFTx2::wr_dat(unsigned char dat)
{
    unsigned short spi_d;
    spi_d =  0x7200 | dat;
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }

    if (spi_port == 0) {    // TFT on SSP0
        LPC_SSP0->DR = spi_d;
        // we have to wait for SPI IDLE to set CS back to high
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI0 not idle
    } else {
        LPC_SSP1->DR = spi_d;
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    }
    _cs0 = 1;
    _cs1 = 1;
}



// the HX8347-D controller do not use the MISO (SDO) Signal.
// This is a bug - ?
// A read will return 0 at the moment

unsigned short SPI_TFTx2::rd_dat (void)
{
    unsigned short val = 0;

    //val = _spi.write(0x73ff);                /* Dummy read 1           */
    //val   = _spi.write(0x0000);              /* Read D8..D15           */
    return (val);
}

void SPI_TFTx2::wr_reg (unsigned char reg, unsigned char val)
{
    wr_cmd(reg);
    wr_dat(val);
}

unsigned short SPI_TFTx2::rd_reg (unsigned char reg)
{
    wr_cmd(reg);
    return(rd_dat());
}

void SPI_TFTx2::tft_reset()
{
    //static unsigned short driverCode;
    _spi.format(16,3);                 // 16 bit spi mode 3
    _spi.frequency(48000000);          // 48 Mhz SPI clock
    _cs0 = 1;                          // cs high
    _cs1 = 1;                          // cs high
    _reset = 0;                        // display reset

    wait_us(50);
    _reset = 1;                       // end reset
    wait_ms(5);

    /* Start Initial Sequence ----------------------------------------------------*/
    wr_reg(0xEA, 0x00);                 /* Reset Power Control 1                */
    wr_reg(0xEB, 0x20);                 /* Power Control 2                      */
    wr_reg(0xEC, 0x0C);                 /* Power Control 3                      */
    wr_reg(0xED, 0xC4);                 /* Power Control 4                      */
    wr_reg(0xE8, 0x40);                 /* Source OPON_N                        */
    wr_reg(0xE9, 0x38);                 /* Source OPON_I                        */
    wr_reg(0xF1, 0x01);                 /*                                      */
    wr_reg(0xF2, 0x10);                 /*                                      */
    wr_reg(0x27, 0xA3);                 /* Display Control 2                    */

    /* Power On sequence ---------------------------------------------------------*/
    wr_reg(0x1B, 0x1B);                 /* Power Control 2                      */
    wr_reg(0x1A, 0x01);                 /* Power Control 1                      */
    wr_reg(0x24, 0x2F);                 /* Vcom Control 2                       */
    wr_reg(0x25, 0x57);                 /* Vcom Control 3                       */
    wr_reg(0x23, 0x8D);                 /* Vcom Control 1                       */

    /* Gamma settings  -----------------------------------------------------------*/
    wr_reg(0x40,0x00);   //
    wr_reg(0x41,0x00);   //
    wr_reg(0x42,0x01);   //
    wr_reg(0x43,0x13);   //
    wr_reg(0x44,0x10);   //
    wr_reg(0x45,0x26);   //
    wr_reg(0x46,0x08);   //
    wr_reg(0x47,0x51);   //
    wr_reg(0x48,0x02);   //
    wr_reg(0x49,0x12);   //
    wr_reg(0x4A,0x18);   //
    wr_reg(0x4B,0x19);   //
    wr_reg(0x4C,0x14);   //
    wr_reg(0x50,0x19);   //
    wr_reg(0x51,0x2F);   //
    wr_reg(0x52,0x2C);   //
    wr_reg(0x53,0x3E);   //
    wr_reg(0x54,0x3F);   //
    wr_reg(0x55,0x3F);   //
    wr_reg(0x56,0x2E);   //
    wr_reg(0x57,0x77);   //
    wr_reg(0x58,0x0B);   //
    wr_reg(0x59,0x06);   //
    wr_reg(0x5A,0x07);   //
    wr_reg(0x5B,0x0D);   //
    wr_reg(0x5C,0x1D);   //
    wr_reg(0x5D,0xCC);   //

    /* Power + Osc ---------------------------------------------------------------*/
    wr_reg(0x18, 0x0036);                 /* OSC Control 1                        */
    wr_reg(0x19, 0x0001);                 /* OSC Control 2                        */
    wr_reg(0x01, 0x0000);                 /* Display Mode Control                 */
    wr_reg(0x1F, 0x0088);                 /* Power Control 6                      */
    wait_ms(5);                           /* Delay 5 ms                           */
    wr_reg(0x1F, 0x0080);                 /* Power Control 6                      */
    wait_ms(5);                           /* Delay 5 ms                           */
    wr_reg(0x1F, 0x0090);                 /* Power Control 6                      */
    wait_ms(5);                           /* Delay 5 ms                           */
    wr_reg(0x1F, 0x00D0);                 /* Power Control 6                      */
    wait_ms(5);                           /* Delay 5 ms                           */

    wr_reg(0x17, 0x0005);                 /* Colmod 16Bit/Pixel                   */

    wr_reg(0x36, 0x0000);                 /* Panel Characteristic                 */
    wr_reg(0x28, 0x0038);                 /* Display Control 3                    */
    wait_ms(40);
    wr_reg(0x28, 0x003C);                 /* Display Control 3                    */
    switch (orientation) {
        case 0:
            wr_reg(0x16, 0x0008);
            break;
        case 1:
            wr_reg(0x16, 0x0068);
            break;
        case 2:
            wr_reg(0x16, 0x00C8);
            break;
        case 3:
            wr_reg(0x16, 0x00A8);
            break;
    }
#if defined USE_DMA
    // setup DMA channel 0
    // Power up the GPDMA.
    LPC_SC->PCONP |= (1UL << 29);
    LPC_GPDMA->DMACConfig = 1;          // enable DMA controller
    // Reset the Interrupt status
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCLLI      = 0;
#endif

    WindowMax ();
}


void SPI_TFTx2::pixel(int x, int y, int color)
{
    unsigned char u,l;
    wr_reg(0x03, (x >> 0));
    wr_reg(0x02, (x >> 8));
    wr_reg(0x07, (y >> 0));
    wr_reg(0x06, (y >> 8));
    wr_cmd(0x22);
    u = color  >> 8;
    l = color & 0xff;
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    if (spi_port == 0) {    // TFT on SSP0
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start Data
        LPC_SSP0->DR = u;           // high byte
        LPC_SSP0->DR = l;           // low byte
        LPC_SSP0->CR0 |= 0x08UL;    // set back to 16 bit
        // we have to wait for SPI IDLE to set CS back to high
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI0 not idle
    } else {
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data
        LPC_SSP1->DR = u;
        LPC_SSP1->DR = l;
        LPC_SSP1->CR0 |= 0x08UL;    // set back to 16 bit
        // we have to wait for SPI IDLE to set CS back to high
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
    }
        _cs0 = 1;
        _cs1 = 1;
}


void SPI_TFTx2::window (unsigned int x, unsigned int y, unsigned int w, unsigned int h)
{
    wr_reg(0x03, x );
    wr_reg(0x02, (x >> 8));
    wr_reg(0x05, x+w-1 );
    wr_reg(0x04, (x+w-1 >> 8));
    wr_reg(0x07,  y );
    wr_reg(0x06, ( y >> 8));
    wr_reg(0x09, ( y+h-1 ));
    wr_reg(0x08, ( y+h-1 >> 8));
}


void SPI_TFTx2::WindowMax (void)
{
    window (0, 0, width(),  height());
}


void SPI_TFTx2::cls (void)
{
    int pixel = ( width() * height());
#if defined USE_DMA
    int dma_count;
    int color = _background;
#endif
    WindowMax();
    wr_cmd(0x22);

    // The SSEL signal is held low until the spi FIFO is emty.
    // We have to lower the SPI clock for the 8 bit start to get the spi running
    // until the next data word
#if defined USE_DMA
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t)&color;
#endif
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    if (spi_port == 0) {    // TFT on SSP0
#if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP0->DR; // we send to SSP0
        /* Enable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x2;
#endif
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start byte
        LPC_SSP0->CR0 |= 0x08UL;    // set to 16 bit
    } else {
#if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP1->DR; // we send to SSP1
        /* Enable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x2;
#endif
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data
        LPC_SSP1->CR0 |= 0x08UL;    // set to 16 bit
    }

#if defined USE_DMA
    // start DMA
    do {
        if (pixel > 4095) {
            dma_count = 4095;
            pixel = pixel - 4095;
        } else {
            dma_count = pixel;
            pixel = 0;
        }
        LPC_GPDMA->DMACIntTCClear = 0x1;
        LPC_GPDMA->DMACIntErrClr = 0x1;
        LPC_GPDMACH0->DMACCControl = dma_count | (1UL << 18) | (1UL << 21) | (1UL << 31) ; // 16 bit transfer , no address increment, interrupt
        LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | (spi_port ? DMA_DEST_SSP1_TX : DMA_DEST_SSP0_TX);
        LPC_GPDMA->DMACSoftSReq = 0x1;   // DMA request

        do {
        } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running

    } while (pixel > 0);
    if (spi_port == 0) {    // TFT on SSP0
        do {
        } while ((0x0010 & LPC_SSP0->SR) == 0x10); // SPI FIFO not empty
        /* disable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x0;
    } else {
        do {
        } while ((0x0010 & LPC_SSP1->SR) == 0x10); // SPI FIFO not empty
        /* disable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x0;
    }
#else  // no DMA
    unsigned int i;
    for (i = 0; i < ( width() * height()); i++)
        _spi.write(_background);
#endif
    _cs0 = 1;
    _cs1 = 1;
}

void SPI_TFTx2::circle(int x0, int y0, int r, int color)
{

    int draw_x0, draw_y0;
    int draw_x1, draw_y1;
    int draw_x2, draw_y2;
    int draw_x3, draw_y3;
    int draw_x4, draw_y4;
    int draw_x5, draw_y5;
    int draw_x6, draw_y6;
    int draw_x7, draw_y7;
    int xx, yy;
    int di;
    //WindowMax();
    if (r == 0) {       /* no radius */
        return;
    }

    draw_x0 = draw_x1 = x0;
    draw_y0 = draw_y1 = y0 + r;
    if (draw_y0 < height()) {
        pixel(draw_x0, draw_y0, color);     /* 90 degree */
    }

    draw_x2 = draw_x3 = x0;
    draw_y2 = draw_y3 = y0 - r;
    if (draw_y2 >= 0) {
        pixel(draw_x2, draw_y2, color);    /* 270 degree */
    }

    draw_x4 = draw_x6 = x0 + r;
    draw_y4 = draw_y6 = y0;
    if (draw_x4 < width()) {
        pixel(draw_x4, draw_y4, color);     /* 0 degree */
    }

    draw_x5 = draw_x7 = x0 - r;
    draw_y5 = draw_y7 = y0;
    if (draw_x5>=0) {
        pixel(draw_x5, draw_y5, color);     /* 180 degree */
    }

    if (r == 1) {
        return;
    }

    di = 3 - 2*r;
    xx = 0;
    yy = r;
    while (xx < yy) {

        if (di < 0) {
            di += 4*xx + 6;
        } else {
            di += 4*(xx - yy) + 10;
            yy--;
            draw_y0--;
            draw_y1--;
            draw_y2++;
            draw_y3++;
            draw_x4--;
            draw_x5++;
            draw_x6--;
            draw_x7++;
        }
        xx++;
        draw_x0++;
        draw_x1--;
        draw_x2++;
        draw_x3--;
        draw_y4++;
        draw_y5++;
        draw_y6--;
        draw_y7--;

        if ( (draw_x0 <= width()) && (draw_y0>=0) ) {
            pixel(draw_x0, draw_y0, color);
        }

        if ( (draw_x1 >= 0) && (draw_y1 >= 0) ) {
            pixel(draw_x1, draw_y1, color);
        }

        if ( (draw_x2 <= width()) && (draw_y2 <= height()) ) {
            pixel(draw_x2, draw_y2, color);
        }

        if ( (draw_x3 >=0 ) && (draw_y3 <= height()) ) {
            pixel(draw_x3, draw_y3, color);
        }

        if ( (draw_x4 <= width()) && (draw_y4 >= 0) ) {
            pixel(draw_x4, draw_y4, color);
        }

        if ( (draw_x5 >= 0) && (draw_y5 >= 0) ) {
            pixel(draw_x5, draw_y5, color);
        }
        if ( (draw_x6 <=width()) && (draw_y6 <= height()) ) {
            pixel(draw_x6, draw_y6, color);
        }
        if ( (draw_x7 >= 0) && (draw_y7 <= height()) ) {
            pixel(draw_x7, draw_y7, color);
        }
    }
    return;
}

void SPI_TFTx2::fillcircle(int x, int y, int r, int color)
{
    int i;
    for (i = 0; i <= r; i++)
        circle(x,y,i,color);
}



void SPI_TFTx2::hline(int x0, int x1, int y, int color)
{
    int w,i;
    w = x1 - x0 + 1;
    window(x0,y,w,1);
    wr_cmd(0x22);
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    if (spi_port == 0) {    // TFT on SSP0
        for (i = 0; i < ( width() * height()); i++)
#if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP0->DR; // we send to SSP0
        /* Enable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x2;
#endif
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start Data
        LPC_SSP0->CR0 |= 0x08UL;    // set to 16 bit
    } else {
#if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP1->DR; // we send to SSP1
        /* Enable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x2;
#endif
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data
        LPC_SSP1->CR0 |= 0x08UL;    // set to 16 bit
    }
#if defined USE_DMA
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t)&color;
    LPC_GPDMACH0->DMACCControl = w | (1UL << 18) | (1UL << 21) | (1UL << 31) ; // 16 bit transfer , no address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | (spi_port ? DMA_DEST_SSP1_TX : DMA_DEST_SSP0_TX);
    LPC_GPDMA->DMACSoftSReq = 0x1;   // start DMA
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
    if (spi_port == 0) {    // TFT on SSP0
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI FIFO not empty
    } else {
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI FIFO not empty
    }
#else
    for (i=0; i<w; i++) {
        _spi.write(color);
    }
    
#endif
    _cs0 = 1;
    _cs1 = 1;
    WindowMax();
    return;
}

void SPI_TFTx2::vline(int x, int y0, int y1, int color)
{
    int h;
    h = y1 - y0 + 1;
    window(x,y0,1,h);
    wr_cmd(0x22);
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    if (spi_port == 0) {    // TFT on SSP0
#if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP0->DR; // we send to SSP0
        /* Enable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x2;
#endif
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start Data
        LPC_SSP0->CR0 |= 0x08UL;    // set to 16 bit
    } else {
#if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP1->DR; // we send to SSP1
        /* Enable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x2;
#endif
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data
        LPC_SSP1->CR0 |= 0x08UL;    // set to 16 bit
    }
#if defined USE_DMA
    LPC_GPDMA->DMACIntTCClear = 0x1;
    LPC_GPDMA->DMACIntErrClr = 0x1;
    LPC_GPDMACH0->DMACCSrcAddr = (uint32_t)&color;
    LPC_GPDMACH0->DMACCControl = h | (1UL << 18) | (1UL << 21) | (1UL << 31) ; // 16 bit transfer , no address increment, interrupt
    LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | (spi_port ? DMA_DEST_SSP1_TX : DMA_DEST_SSP0_TX);
    LPC_GPDMA->DMACSoftSReq = 0x1;
    do {
    } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running

    if (spi_port == 0) {    // TFT on SSP0
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI FIFO not empty
    } else {
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI FIFO not empty
    }
#else
    for (int y=0; y<h; y++) {
        _spi.write(color);
    }
#endif
    _cs0 = 1;
    _cs1 = 1;
    WindowMax();
    return;
}

void SPI_TFTx2::line(int x0, int y0, int x1, int y1, int color)
{
    //WindowMax();
    int   dx = 0, dy = 0;
    int   dx_sym = 0, dy_sym = 0;
    int   dx_x2 = 0, dy_x2 = 0;
    int   di = 0;

    dx = x1-x0;
    dy = y1-y0;

    // vline and hline are very slow for some reason so not using
    //if (dx == 0) {        /* vertical line */
    //    if (y1 > y0) vline(x0,y0,y1,color);
    //    else vline(x0,y1,y0,color);
    //    return;
    //}

    if (dx > 0) {
        dx_sym = 1;
    } else {
        dx_sym = -1;
    }
    //if (dy == 0) {        /* horizontal line */
    //    if (x1 > x0) hline(x0,x1,y0,color);
    //    else  hline(x1,x0,y0,color);
    //    return;
    //}

    if (dy > 0) {
        dy_sym = 1;
    } else {
        dy_sym = -1;
    }

    dx = dx_sym*dx;
    dy = dy_sym*dy;

    dx_x2 = dx*2;
    dy_x2 = dy*2;

    if (dx >= dy) {
        di = dy_x2 - dx;
        while (x0 != x1) {

            pixel(x0, y0, color);
            x0 += dx_sym;
            if (di<0) {
                di += dy_x2;
            } else {
                di += dy_x2 - dx_x2;
                y0 += dy_sym;
            }
        }
        pixel(x0, y0, color);
    } else {
        di = dx_x2 - dy;
        while (y0 != y1) {
            pixel(x0, y0, color);
            y0 += dy_sym;
            if (di < 0) {
                di += dx_x2;
            } else {
                di += dx_x2 - dy_x2;
                x0 += dx_sym;
            }
        }
        pixel(x0, y0, color);
    }
    return;
}


void SPI_TFTx2::rect(int x0, int y0, int x1, int y1, int color)
{

    if (x1 > x0) hline(x0,x1,y0,color);
    else  hline(x1,x0,y0,color);

    if (y1 > y0) vline(x0,y0,y1,color);
    else vline(x0,y1,y0,color);

    if (x1 > x0) hline(x0,x1,y1,color);
    else  hline(x1,x0,y1,color);

    if (y1 > y0) vline(x1,y0,y1,color);
    else vline(x1,y1,y0,color);

    return;
}



void SPI_TFTx2::fillrect(int x0, int y0, int x1, int y1, int color)
{

    int h = y1 - y0 + 1;
    int w = x1 - x0 + 1;
    int pixel = h * w;
    int dma_count;
    window(x0,y0,w,h);
    wr_cmd(0x22);
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    if (spi_port == 0) {    // TFT on SSP0
        #if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP0->DR; // we send to SSP0
        /* Enable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x2;
        #endif
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start Data
        LPC_SSP0->CR0 |= 0x08UL;    // set to 16 bit
    } else {
        #if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP1->DR; // we send to SSP1
        /* Enable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x2;
        #endif
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data
        LPC_SSP1->CR0 |= 0x08UL;    // set to 16 bit
    }
    #if defined USE_DMA
    do {
        if (pixel > 4095) {
            dma_count = 4095;
            pixel = pixel - 4095;
        } else {
            dma_count = pixel;
            pixel = 0;
        }
        LPC_GPDMA->DMACIntTCClear = 0x1;
        LPC_GPDMA->DMACIntErrClr = 0x1;
        LPC_GPDMACH0->DMACCSrcAddr = (uint32_t)&color;
        LPC_GPDMACH0->DMACCControl = dma_count | (1UL << 18) | (1UL << 21) | (1UL << 31) ; // 16 bit transfer , no address increment, interrupt
        LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | (spi_port ? DMA_DEST_SSP1_TX : DMA_DEST_SSP0_TX);
        LPC_GPDMA->DMACSoftSReq = 0x1;
        do {
        } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running

    } while (pixel > 0);

    if (spi_port == 0) {    // TFT on SSP0
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI FIFO not empty
    } else {
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI FIFO not empty
    }
   
    #else  // no DMA 
    for (int p=0; p<pixel; p++) {
        _spi.write(color);
    }
    #endif

    _cs0 = 1;
    _cs1 = 1;
    WindowMax();
    return;
}


void SPI_TFTx2::locate(int x, int y)
{
    char_x = x;
    char_y = y;
}



int SPI_TFTx2::columns()
{
    return width() / font[1];
}



int SPI_TFTx2::rows()
{
    return height() / font[2];
}



int SPI_TFTx2::_putc(int value)
{
    if (value == '\n') {    // new line
        char_x = 0;
        char_y = char_y + font[2];
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    } else {
        character(char_x, char_y, value);
    }
    return value;
}




void SPI_TFTx2::character(int x, int y, int c)
{
    unsigned int hor,vert,offset,bpl,j,i,b,p;
    unsigned char* zeichen;
    unsigned char z,w;
    #if defined USE_DMA
    unsigned int pixel;
    unsigned int dma_count,dma_off;
    uint16_t *buffer;
    #endif

    if ((c < 31) || (c > 127)) return;   // test char range

    // read font parameter from start of array
    offset = font[0];                    // bytes / char
    hor = font[1];                       // get hor size of font
    vert = font[2];                      // get vert size of font
    bpl = font[3];                       // bytes per line

    if (char_x + hor > width()) {
        char_x = 0;
        char_y = char_y + vert;
        if (char_y >= height() - font[2]) {
            char_y = 0;
        }
    }
    window(char_x, char_y,hor,vert); // char box
    wr_cmd(0x22);

#if defined USE_DMA
    pixel = hor * vert;  // calculate buffer size

    buffer = (uint16_t *) malloc (2*pixel); // we need a buffer for the 16 bit
    if (buffer == NULL) {
        //led = 1;
        //pc.printf("Malloc error !\n\r");
        return;         // error no memory
    }

    zeichen = &font[((c -32) * offset) + 4]; // start of char bitmap
    w = zeichen[0];                          // width of actual char
    p = 0;
    // construct the char into the buffer
    for (j=0; j<vert; j++) {  //  vert line
        for (i=0; i<hor; i++) {   //  horz line
            z =  zeichen[bpl * i + ((j & 0xF8) >> 3)+1];
            b = 1 << (j & 0x07);
            if (( z & b ) == 0x00) {
                buffer[p] = _background;
            } else {
                buffer[p] = _foreground;
            }
            p++;
        }
    }

    // copy the buffer with DMA SPI to display
    dma_off = 0;  // offset for DMA transfer
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    if (spi_port == 0) {    // TFT on SSP0
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP0->DR; // we send to SSP0
        /* Enable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x2;
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start Data
        LPC_SSP0->CR0 |= 0x08UL;    // set to 16 bit
    } else {
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP1->DR; // we send to SSP1
        /* Enable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x2;
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data
        LPC_SSP1->CR0 |= 0x08UL;    // set to 16 bit
    }

    // start DMA
    do {
        if (pixel > 4095) {         // this is a giant font !
            dma_count = 4095;
            pixel = pixel - 4095;
        } else {
            dma_count = pixel;
            pixel = 0;
        }
        LPC_GPDMA->DMACIntTCClear = 0x1;
        LPC_GPDMA->DMACIntErrClr = 0x1;
        LPC_GPDMACH0->DMACCSrcAddr = (uint32_t) (buffer + dma_off);
        LPC_GPDMACH0->DMACCControl = dma_count | (1UL << 18) | (1UL << 21) | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 16 bit transfer , address increment, interrupt
        LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | (spi_port ? DMA_DEST_SSP1_TX : DMA_DEST_SSP0_TX);
        LPC_GPDMA->DMACSoftSReq = 0x1;
        do {
        } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
        dma_off = dma_off + dma_count;
    } while (pixel > 0);

    free ((uint16_t *) buffer);

    if (spi_port == 0) {    // TFT on SSP0
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI0 not idle
        /* disable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x0;
    } else {
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI1 not idle
        /* disable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x0;
    }
#else
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    if (spi_port == 0) {    // TFT on SSP0
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start Data
        LPC_SSP0->CR0 |= 0x08UL;    // set to 16 bit
    } else {
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data
        LPC_SSP1->CR0 |= 0x08UL;    // set to 16 bit
    }
    zeichen = &font[((c -32) * offset) + 4]; // start of char bitmap
    w = zeichen[0];                          // width of actual char
     for (j=0; j<vert; j++) {  //  vert line
        for (i=0; i<hor; i++) {   //  horz line
            z =  zeichen[bpl * i + ((j & 0xF8) >> 3)+1];
            b = 1 << (j & 0x07);
            if (( z & b ) == 0x00) {
                _spi.write(_background);
            } else {
                _spi.write(_foreground);
            }
        }
    }
#endif    
    _cs0 = 1;
    _cs1 = 1;
    WindowMax();
    if ((w + 2) < hor) {                   // x offset to next char
        char_x += w + 2;
    } else char_x += hor;

}


void SPI_TFTx2::set_font(unsigned char* f)
{
    font = f;
}

void SPI_TFTx2::Bitmap(unsigned int x, unsigned int y, unsigned int w, unsigned int h,unsigned char *bitmap)
{
    unsigned int  j;
    int padd;
    unsigned short *bitmap_ptr = (unsigned short *)bitmap;
    // the lines are padded to multiple of 4 bytes in a bitmap
    padd = -1;
    do {
        padd ++;
    } while (2*(w + padd)%4 != 0);
    window(x, y, w, h);
    wr_cmd(0x22);
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }
    if (spi_port == 0) {    // TFT on SSP0
#if defined USE_DMA    
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP0->DR; // we send to SSP0
        /* Enable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x2;
#endif        
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start Data
        LPC_SSP0->CR0 |= 0x08UL;    // set to 16 bit

    } else {
#if defined USE_DMA    
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP1->DR; // we send to SSP1
        /* Enable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x2;   
#endif        
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data command
        LPC_SSP1->CR0 |= 0x08UL;    // set to 16 bit
    }

    bitmap_ptr += ((h - 1)* (w + padd));
#if defined USE_DMA      
    for (j = 0; j < h; j++) {        //Lines
        LPC_GPDMA->DMACIntTCClear = 0x1;
        LPC_GPDMA->DMACIntErrClr = 0x1;
        LPC_GPDMACH0->DMACCSrcAddr = (uint32_t)bitmap_ptr;
        LPC_GPDMACH0->DMACCControl = w | (1UL << 18) | (1UL << 21) | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 16 bit transfer , address increment, interrupt
        LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | (spi_port ? DMA_DEST_SSP1_TX : DMA_DEST_SSP0_TX);
        LPC_GPDMA->DMACSoftSReq = 0x1;
        do {
        } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running

        bitmap_ptr -= w;
        bitmap_ptr -= padd;
    }
#else
    unsigned int i;
    for (j = 0; j < h; j++) {        //Lines
        for (i = 0; i < w; i++) {     // copy pixel data to TFT
            _spi.write(*bitmap_ptr);    // one line
            bitmap_ptr++;
        }
        bitmap_ptr -= 2*w;
        bitmap_ptr -= padd;
    }   
#endif    
    if (spi_port == 0) {    // TFT on SSP0
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI FIFO not empty
    } else {
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI FIFO not empty
    }
    _cs0 = 1;
    _cs1 = 1;
    WindowMax();
}


int SPI_TFTx2::BMP_16(unsigned int x, unsigned int y, const char *Name_BMP)
{

#define OffsetPixelWidth    18
#define OffsetPixelHeigh    22
#define OffsetFileSize      34
#define OffsetPixData       10
#define OffsetBPP           28

    char filename[50];
    unsigned char BMP_Header[54];
    unsigned short BPP_t;
    unsigned int PixelWidth,PixelHeigh,start_data;
    unsigned int    i,off;
    int padd,j;
    unsigned short *line;

    // get the filename
    LocalFileSystem local("local");
    sprintf(&filename[0],"/local/");
    i=7;
    while (*Name_BMP!='\0') {
        filename[i++]=*Name_BMP++;
    }

    fprintf(stderr, "filename : %s \n\r",filename);

    FILE *Image = fopen((const char *)&filename[0], "rb");  // open the bmp file
    if (!Image) {
        return(0);      // error file not found !
    }

    fread(&BMP_Header[0],1,54,Image);      // get the BMP Header

    if (BMP_Header[0] != 0x42 || BMP_Header[1] != 0x4D) {  // check magic byte
        fclose(Image);
        return(-1);     // error no BMP file
    }

    BPP_t = BMP_Header[OffsetBPP] + (BMP_Header[OffsetBPP + 1] << 8);
    if (BPP_t != 0x0010) {
        fclose(Image);
        return(-2);     // error no 16 bit BMP
    }

    PixelHeigh = BMP_Header[OffsetPixelHeigh] + (BMP_Header[OffsetPixelHeigh + 1] << 8) + (BMP_Header[OffsetPixelHeigh + 2] << 16) + (BMP_Header[OffsetPixelHeigh + 3] << 24);
    PixelWidth = BMP_Header[OffsetPixelWidth] + (BMP_Header[OffsetPixelWidth + 1] << 8) + (BMP_Header[OffsetPixelWidth + 2] << 16) + (BMP_Header[OffsetPixelWidth + 3] << 24);
    if (PixelHeigh > height() + y || PixelWidth > width() + x) {
        fclose(Image);
        return(-3);      // to big
    }

    start_data = BMP_Header[OffsetPixData] + (BMP_Header[OffsetPixData + 1] << 8) + (BMP_Header[OffsetPixData + 2] << 16) + (BMP_Header[OffsetPixData + 3] << 24);

    line = (unsigned short *) malloc (2 * PixelWidth); // we need a buffer for a line
    if (line == NULL) {
        return(-4);         // error no memory
    }

    // the bmp lines are padded to multiple of 4 bytes
    padd = -1;
    do {
        padd ++;
    } while ((PixelWidth * 2 + padd)%4 != 0);


//fseek(Image, 70 ,SEEK_SET);
    window(x, y,PixelWidth ,PixelHeigh);
    wr_cmd(0x22);
    if (seldisp == 0) {
        _cs0 = 0;
        _cs1 = 1;
    } else if (seldisp == 1) {
        _cs0 = 1;
        _cs1 = 0;
    } else {
        _cs0 = 0;
        _cs1 = 0;
    }

    if (spi_port == 0) {    // TFT on SSP0
#if defined USE_DMA
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP0->DR; // we send to SSP0
        /* Enable SSP0 for DMA. */
        LPC_SSP0->DMACR = 0x2;
#endif
        LPC_SSP0->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP0->DR = 0x72;        // start Data
        LPC_SSP0->CR0 |= 0x08UL;    // set to 16 bit

    } else {
#if defined USE_DMA    
        LPC_GPDMACH0->DMACCDestAddr = (uint32_t)&LPC_SSP1->DR; // we send to SSP1
        /* Enable SSP1 for DMA. */
        LPC_SSP1->DMACR = 0x2;
#endif
        LPC_SSP1->CR0 &= ~(0x08UL); // set to 8 bit
        LPC_SSP1->DR = 0x72;        // start Data
        LPC_SSP1->CR0 |= 0x08UL;    // set to 16 bit     
    }
    for (j = PixelHeigh - 1; j >= 0; j--) {               //Lines bottom up
        off = j * (PixelWidth  * 2 + padd) + start_data;   // start of line
        fseek(Image, off ,SEEK_SET);
        fread(line,1,PixelWidth * 2,Image);       // read a line - slow !
#if defined USE_DMA
        LPC_GPDMA->DMACIntTCClear = 0x1;
        LPC_GPDMA->DMACIntErrClr = 0x1;
        LPC_GPDMACH0->DMACCSrcAddr = (uint32_t)line;
        LPC_GPDMACH0->DMACCControl = PixelWidth | (1UL << 18) | (1UL << 21) | (1UL << 31) |  DMA_CHANNEL_SRC_INC ; // 16 bit transfer , address increment, interrupt
        LPC_GPDMACH0->DMACCConfig  = DMA_CHANNEL_ENABLE | DMA_TRANSFER_TYPE_M2P | (spi_port ? DMA_DEST_SSP1_TX : DMA_DEST_SSP0_TX);
        LPC_GPDMA->DMACSoftSReq = 0x1;
        do {
        } while ((LPC_GPDMA->DMACRawIntTCStat & 0x01) == 0); // DMA is running
#else
    for (i = 0; i < PixelWidth; i++) {        // copy pixel data to TFT
            _spi.write(line[i]);                  // one 16 bit pixel
        } 
#endif            
    }
    if (spi_port == 0) {    // TFT on SSP0
        do {
        } while ((LPC_SSP0->SR & 0x10) == 0x10); // SPI FIFO not empty
    } else {
        do {
        } while ((LPC_SSP1->SR & 0x10) == 0x10); // SPI FIFO not empty
    }
    _cs0 = 1;
    _cs1 = 1;
    free (line);
    fclose(Image);
    WindowMax();
    return(1);
}