/* mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
#include <string.h>
#include "mbed.h"
#include "BLE.h"
 
#include "UARTService.h"
#include "BatteryService.h"
#include "DeviceInformationService.h" 
static const uint16_t uuid16_list[] = {GattService::UUID_BATTERY_SERVICE,
                                GattService::UUID_DEVICE_INFORMATION_SERVICE};

Serial pc(USBTX, USBRX);
BLEDevice  ble;

//Used to access the defined in main UARTService object globally.
UARTService *uart;

//Keeping a receive buffer to alter the data received.
uint8_t DatatoSend[1000];
 
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    pc.printf("\n Disconnected.!!\n");
    pc.printf("\n Restarting the advertising process.\n");
    ble.startAdvertising();
}
 
void periodicCallback(void)
{
}

void onDataWritten(const GattWriteCallbackParams *params)
{
    /*if received something, print it out on PC screen*/
    if ((uart != NULL) && (params->handle == uart->getTXCharacteristicHandle())) {
        uint16_t bytesRead = params->len;
 
        pc.printf("Data Received\n");
        /*for all received data, send a copy back to the BLE central(in this case the phone app)*/
        for(int j=0;j<bytesRead;j++)
        {
            pc.printf(" %x\n",*((params->data)+j));
            DatatoSend[j]=(*((params->data)+j));
        }
        
        wait(1);
        ble.updateCharacteristicValue(uart->getRXCharacteristicHandle(), DatatoSend, bytesRead);
        //Use the below statement for loopback.
        //ble.updateCharacteristicValue(uart->getRXCharacteristicHandle(), params->data, bytesRead);
        
        /*print out what have just been sent*/
        pc.printf("Data Sent\n");
        for(int j=0;j<bytesRead;j++)
        {
            pc.printf(" %x\n",DatatoSend[j]);
        }
        wait(1);
    }
}

int main(void)
{
    
    pc.baud(9600);
    pc.printf("Hello, starting BlueNRG Serial Console Demo!\n");
    Ticker ticker;
    ticker.attach(periodicCallback, 1);
 
    pc.printf("Initialising the module\n\r");
    ble.init();
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(onDataWritten);
    
    UARTService uartService(ble);
    uart = &uartService;
    
    BatteryService  battery(ble);
    DeviceInformationService deviceInfo(ble, "ARM", "Model1", "SN1", "hw-rev1", "fw-rev1", "soft-rev1");
 
    /* setup advertising */
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::SHORTENED_LOCAL_NAME,
                                     (const uint8_t *)"BLE UART", sizeof("BLE UART") - 1);
    ble.accumulateAdvertisingPayload(GapAdvertisingData::INCOMPLETE_LIST_128BIT_SERVICE_IDS,
                                     (const uint8_t *)UARTServiceUUID_reversed, sizeof(UARTServiceUUID_reversed));
    ble.accumulateAdvertisingPayload(GapAdvertisingData::INCOMPLETE_LIST_16BIT_SERVICE_IDS, (uint8_t *)uuid16_list, sizeof(uuid16_list));                               
    ble.setAdvertisingInterval(160); /* 100ms; in multiples of 0.625ms. */
    ble.startAdvertising();
    pc.printf("Start Advertising.\n");
 
    while (true) {
        ble.waitForEvent();
    }
}
