#ifndef MBED_AD7303_H
#define MBED_AD7303_H
/** Classe AD7303.
 *  Utilisée par l'IOGS pour le DAC AD7303.
 *
 * Example:
 * @code
 * #include "mbed.h"
 * #include "AD7303.h"
 * 
 * AD7303 dac8b(D10,D11,D12,D13);
 * char c;
 * 
 * int main()
 * {
 *     dac8b.initDAC();
 *     while(1) {
 *         c=' ';
 *         for(c=0; c<255; c++) {
 *             dac8b.changeDACA_IntRef(c);
 *         }
 *     }
 * }
 * @endcode
 */
#include "mbed.h"
/* Modes de fonctionnement */
#define     INT_REF         0x00        // Internal reference voltage Vref ? VDD/2
#define     EXT_REF         0x80        // External reference voltage

#define     POWER_DOWN_DACA 0b00001000  // Switch off DACA
#define     POWER_DOWN_DACB 0b00010000  // Switch off DACB

#define     UPDATE_DACA     0b00000011  // Update DACA from shift register
#define     UPDATE_DACB     0b00100111  // Update DACB from shitf register

class AD7303{
public:
    /** Déclare un CNA AD7303
         *
         * @param Cs Broche de la carte Nucleo L476RG reliée à la broche Cs du CNA AD7303
         * @param Miso Broche de la carte Nucleo L476RG reliée à la broche Miso du CNA AD7303
         * @param Mosi Broche de la carte Nucleo L476RG reliée à la broche Mosi du CNA AD7303
         * @param Sck Broche de la carte Nucleo L476RG reliée à la broche Sck du CNA AD7303

         */
    AD7303(PinName Cs,PinName Mosi, PinName Miso, PinName Sck);
    /** Fonction d'initialisation du CNA AD7303
         * Internal Vref / Power-Up on both DACA and DACB
         * Load DACA and DACB with VDD/2 on outputs
         * @note l'appel de cette fonction est essentielle pour utiliser le CNA AD7303
         */
    void initDAC(void);
    /** Changement mode de configuration
         *
         * @param mode mode de fonctionnement souhaite
         * @param val valeur en sortie (8 bits)
         * @returns rien
         */
    void changeModeDAC(char mode, char val);
        /** Modification de la tension de sortie du DACA - Reference interne
         *
         * @param val valeur sur 8 bits de la tension a appliquer
         * @returns rien
         */
    void changeDACA_IntRef(char val);
        /** Modification de la tension de sortie du DACB - Reference interne
         *
         * @param val valeur sur 8 bits de la tension a appliquer
         * @returns rien
         */
    void changeDACB_IntRef(char val);
        /** Modification de la tension de sortie du DACA - Reference externe
         *
         * @param val valeur sur 8 bits de la tension a appliquer
         * @returns rien
         */
    void changeDACA_ExtRef(char val);
        /** Modification de la tension de sortie du DACB - Reference externe
         *
         * @param val valeur sur 8 bits de la tension a appliquer
         * @returns rien
         */
    void changeDACB_ExtRef(char val);
private:
    DigitalOut CS_DAC; //CS
    SPI ad7303; // mosi, miso, sclk
};
 
#endif