/*
   RFDecoder - Ported from the Lucas Maziero Arduino libary to decode RF controls based on chip HT6P20
  Copyright (c) 2011 Miguel Maldonado.  All right reserved.
  
  Project home: https://github.com/lucasmaziero/RF_HT6P20

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
*/

#include "RFDecoder.h"
#include "mbed.h"
#include <Pulse.h>


/** Class constructor.
 * The constructor assigns the specified pinout, attatches 
 * an Interrupt to the receive pin. 
 * @param tx Transmitter pin of the RF module.
 * @param rx Receiver pin of the RF module.
 */
 
 RFDecoder::RFDecoder(PinName tx, PinName rx) : _tx(DigitalOut(tx)), _rx(PulseInOut(rx)){
     RFDecoder::addressFull = NULL;     
     
     }
 
unsigned long RFDecoder::getCode(){
    return(addressFull);
} 

bool RFDecoder::available(){
     static bool startbit=0;          //checks if start bit was identified
     static int counter;            //received bits counter: 20 of Address + 4 of Data + 4 of EndCode (Anti-Code)
     static unsigned long buffer;   //buffer for received data storage
     
     int dur0, dur1;                // pulses durations (auxiliary)
        
     if (!startbit) {               // Check the PILOT CODE until START BIT;
     dur0 = _rx.read_low_us();
     
        //If time at "0" is between 9200 us (23 cycles of 400us) and 13800 us (23 cycles of 600 us).
        if((dur0 > 9200) && (dur0 < 13800) && !startbit){
            //calculate wave length - lambda
            lambda_RX = (dur0 / 23);
            
            //Reset variables
            dur0 = 0;
            buffer = 0;
            counter = 0;

            startbit = true;    
        }
     }
     
     //If Start Bit is OK, then starts measure os how long the signal is level "1" and check is value is into acceptable range.
     if (startbit && counter < 28){
         ++counter;

         dur1 = _rx.read_high_us();

        if((dur1 > 0.5 * lambda_RX) && (dur1 < (1.5 * lambda_RX))){  //If pulse width at "1" is between "0.5 and 1.5 lambda", means that pulse is only one lambda, so the data is "1".     
          buffer = (buffer << 1) + 1;   // add "1" on data buffer
        }
        else if((dur1 > 1.5 * lambda_RX) && (dur1 < (2.5 * lambda_RX))){  //If pulse width at "1" is between "1.5 and 2.5 lambda", means that pulse is two lambdas, so the data is "0".
          buffer = (buffer << 1);       // add "0" on data buffer
        }
        else{
          startbit = false; //Reset the loop
        }
      }
      
      //Check if all 28 bits were received (20 of Address + 4 of Data + 4 of Anti-Code)
      if (counter==28) 
      { 
      
        // Check if Anti-Code is OK (last 4 bits of buffer equal "0101")
        if ((bitRead(buffer, 0) == 1) && (bitRead(buffer, 1) == 0) && (bitRead(buffer, 2) == 1) && (bitRead(buffer, 3) == 0)){     
          counter = 0;
          startbit = false;
    
          //Get ADDRESS COMPLETO from Buffer
          addressFull = buffer;
          
          //If a valid data is received, return OK
          return true;
        }
        else
        {
          //Reset the loop
          startbit = false;
        }
    
      }
      
      //If none valid data is received, return NULL and FALSE values 
      addressFull = 0; 
      return false;       
}

bool RFDecoder::bitRead( int N, int pos)
   {                            // pos = 7 6 5 4 3 2 1 0
       int b = N >> pos ;       // Shift bits
       b = b & 1 ;              // get only the last bit
       return b ;
   }   
     