#include "ESP01.h"

// Constructor
ESP01::ESP01(PinName tx, PinName rx, int br) : comm(tx, rx)
{
    comm.baud(br);
}

// Destructor
ESP01::~ESP01() { }

// Add <CR> + <LF> at the end of the string
void ESP01::AddEOL(char * s)
{
    char k   = strlen(s); // Finds position of NULL character
    s[k]     = '\r';    // switch NULL for <CR>
    s[k + 1] = '\n';    // Add <LF>
    s[k + 2] = '\0';    // Add NULL at the end
}

// Add one ASCII character at the end of the string
void ESP01::AddChar(char * s, char c)
{
    char k   = strlen(s);
    s[k]     = c;
    s[k + 1] = '\0';
}

// Sends command to ESP01. Receives the command string
void ESP01::SendCMD(char* s)
{
    comm.printf("%s%s", s, CMD_END);
}

// Resets the ESP01
void ESP01::Reset(void)
{
    SendCMD("AT+RST");
}

// Receive reply until no character is received after a given timeout in miliseconds
bool ESP01::RcvReply(char* r, int to)
{
    Timer t;
    bool ended = 0;
    char c;
    
    strcpy(r, "");
    t.start();
    while(!ended) {
        if(comm.readable()) {
            c = comm.getc();
            AddChar(r, c);
            t.start();
        }
        if(t.read_ms() > to) {
                ended = 1;
        }
    }
    
    AddChar(r, '\0');
    return r[0] != '\0';
}

// Gets the AP list. Parameter: the string to receive the list
bool ESP01::GetList(char *l)
{
    SendCMD("AT+CWLAP");
    return RcvReply(l, 5000); // Needs big timeout because it takes long to start replying
}

// Joins a Wifi AP. Parameters: SSID and Password (strings)
void ESP01::Join(char *id, char *pwd)
{
    char cmd[255];
    sprintf(cmd, "AT+CWJAP=\"%s\",\"%s\"", id, pwd);
    SendCMD(cmd);
}

// Gets ESP IP. Parameter: string to contain IP
bool ESP01::GetIP(char *ip)
{
    SendCMD("AT+CIFSR");
    return RcvReply(ip, 3000);
}

//Defines wifi mode; Parameter: mode; 1= STA, 2= AP, 3=both
void ESP01::SetMode(WiFiMode mode)
{
    char cmd[15];
    strcpy(cmd, "AT+CWMODE=");
    AddChar(cmd, mode + '0'); // // Converts number into corresponding ASCII character
    SendCMD(cmd);
}

// Quits the AP
void ESP01::Quit(void)
{
    SendCMD("AT+CWQAP");
}

// Sets single connection
void ESP01::SetSingle(void)
{
    SendCMD("AT+CIPMUX=0");
}

// Sets multiple connection
void ESP01::SetMultiple(void)
{
    SendCMD("AT+CIPMUX=1");
}

// Gets connection status. Parameter: string to contain status
bool ESP01::GetConnStatus(char * st)
{
    SendCMD("AT+CIPSTATUS");
    return RcvReply(st, 2000);
}

// Starts server mode. Parameter: port to be used
void ESP01::StartServerMode(int port)
{
    char rs[25];
    sprintf(rs, "AT+CIPSERVER=1,%d", port);
    SendCMD(rs);
}

// Close server mode.
void ESP01::CloseServerMode(void)
{
    SendCMD("AT+CIPSERVER=0");
}

void ESP01::setTransparent(void)
{
    SendCMD("AT+CIPMODE=0");
}

void ESP01::startTCPConn(char *IP, int port)
{
    char rs[100];
    sprintf(rs, "AT+CIPSTART=0,\"TCP\",\"%s\",%d", IP, port);
    SendCMD(rs);
}

void ESP01::sendURL(char *URL, char* IP, char *command)
{
    char snd[20], http_cmd[300];

    sprintf(http_cmd, "GET %s HTTP/1.1%sHost: %s%sConnection: close%s%s", URL, CMD_END, IP, CMD_END, CMD_END, CMD_END);
    sprintf(snd,"AT+CIPSEND=0,%d",strlen(http_cmd));
    strcpy(command, http_cmd);

    SendCMD(snd);
    wait(1);
    SendCMD(http_cmd);
}

void ESP01::deepsleep(size_t ms)
{
    char snd[20];
    sprintf(snd, "AT+GSLP=%d", ms);
    SendCMD(snd);
}

//------------------   FUNCIONES QUE YO HE AGREGADO  ------------------//
// Starts SmartConfig
void ESP01::StartSmartConfig(void)
{
    SendCMD("AT+CWSTARTSMART=1");
}

//Disable Echo
void ESP01::DisableEcho(void)
{
    SendCMD("ATE0");
}

//Enable DHCP
void ESP01::EnableDHCP(void)
{
    SendCMD("AT+CWDHCP=1,1");
}

// Receive reply until no character is received after a given timeout in miliseconds
void ESP01::RcvSingleReply(char * r)
{
    bool ended = 0;
    char c;
    Timer t;

    strcpy(r, "");
    t.start();
    while(!ended) {
        if(comm.readable()) {
            c = comm.getc();
            AddChar(r, c);
        }
        if((c==0x0D)||(t.read_ms() > 5000)) {
            ended = 1;
        }
    }
    //AddChar(r, 0x00);
}

// Gets connection status code. Parameter: string to contain status
void ESP01::GetConnStatusCode(char * st)
{
    char cmd[25];
    strcpy(cmd, "AT+CIPSTATUS");
    SendCMD(cmd);
    RcvSingleReply(st);
}

// Set the Maximum Connections Allowed by Server. Parameter: maximum number of connections
void ESP01::SetMaxTCPConn(int maxconn)
{
    char rs[50];
    sprintf(rs, "AT+CIPSERVERMAXCONN=%d", maxconn);
    SendCMD(rs);
}

// Returns true if data has been received. Parameter: string to contain data received
bool ESP01::TCPDataAvailable(char *data)
{
    return RcvReply(data, 500);
}

// Send TCP Data. Parameter:
void ESP01::SendTCPData(int socket,int len, char *data)
{
    char i;
    char rs[50];
    sprintf(rs, "AT+CIPSEND=%d,%d",socket,len);
    SendCMD(rs);
    wait(1);
    SendCMD(data);   
}