/**
* @file WorldBuilder.h
* @brief Header file containing the world generation member functions and variables
* @brief Revision 1.0
*
* @author Thomas Barnaby Gill
* @date   11th May 2015
*/

#include "mbed.h"

#ifndef WORLDBUILDER_H
#define WORLDBUILDER_H

#define WALL 0
#define FLOOR 1
#define ENTER 2
#define EXIT 3
#define FLOOR_SEEN 4
#define CHEST 5
#define CHEST_OPENED 6

#define RIGHT 0
#define LEFT 1
#define UP 2
#define DOWN 3

#define MAP_WIDTH 84
#define MAP_HEIGHT 48

/** Map Array
*
* @param map - Array which stores the map data for the game
*
*/
extern int map[MAP_WIDTH][MAP_HEIGHT];

//Enterance coordinates
/** Enterance x co-ordinate
*
* @param enx - Variable which stores the x co-ordinate of the enterance of the current level
*
*/
extern int enx;

/** Enterance y co-ordinate
*
* @param eny - Variable which stores the y co-ordinate of the enterance of the current level
*
*/
extern int eny;

//Exit coordinates
/** Exit x co-ordinate
*
* @param exx - Variable which stores the x co-ordinate of the exit of the current level
*
*/
extern int exx;

/** Exit y co-ordinate
*
* @param exy- Variable which stores the y co-ordinate of the exit of the current level
*
*/
extern int exy;

/** Start x co-ordinate
*
* @param sx - Variable which stores the x co-ordinate of the start of a room
*
*/
extern int sx;

/** Start y co-ordinate
*
* @param sy - Variable which stores the y co-ordinate of the start of a room
*
*/
extern int sy;

/** Current Level
*
* @param level - Variable which stores the level number the player is currently on
*
*/
extern int level;

/** Player x co-ordinate
*
* @param px - Variable which stores the current x co-ordinate of the player
*
*/
extern int px;
/** Player y co-ordinate
*
* @param py - Variable which stores the current x co-ordinate of the player
*
*/
extern int py;

/** Wall Fill
*
* Fills the map array with walls
*
*/
void Walls();

/** First room Creator
*
* Creates a room at a random position in the map with an enterance in it
*
*/
void FirstRoom();

/** Exit room Creator
*
* Creates a room at a random position in the map with an exit in it
*
*/
void ExitRoom();

/** Dungeon Room Creator
*
* Creates a room at a random position in the map with a chance of having a chest in it
*
*/
void DungeonRoomBuilder();

/** Neighbours
*
* Checks how many neighbouring cells at the current position are floors. It returns an integer value equal to the number of neighbours with floors.
* @param i - x co-ordinate of current position
* @param j - y co-ordinate of current position
*
*/
int Neighbours(int i, int j);

/** Dead Ends
*
* Deletes corridors that are dead ends
* @param d - How many times the function should iterate through the map
*
*/
void DeadEnds(int d);

/** Border
*
* Creates a 1 cell thick border of walls at the extremeties of the map
*
*/
void Border();

/** Random Floor Setter
*
* Randomly turns a wall in the map into a floor.
* @pararm r - How many times the function should be iterated
*
*/
void RandFloor(int r);

/** Maze Kill
*
* The 'kill' phase of a hunt and kill maze algorithm.
* The algorithm walks in random directions (which do not conflict with previously made paths), turning walls into floors in the current direction until it hits a dead end
*
*/
void MazeKill();

/** Maze Maker
*
* Implements a 'hunt and kill' perfect maze algorithm
*
* (See http://www.astrolog.org/labyrnth/algrithm.htm for more information on the algorithm)
*
*/
void Maze();

/** Dungeon Builder
*
* Creates a 'dungeon' style level on the map array by generating a maze, adding enterance, exit and filler rooms, turning random tiles into floors and then removing the dead ends
*
*/
void DungeonBuilder();

/** Labyrinth Builder
*
* Creates a labyrinth level on the map array by generating a maze, adding enterance and exit rooms, turning random tiles into floors and then removing some of the dead ends
*
*/
void LabyrinthBuilder();

/** World Generator
*
* Clears the current map using the Walls() function and then generates a dungeon or labyrinth level
*
*/
void World();

#endif