/**
* @file Game.h
* @brief Header file containing the main game member functions and variables
* @brief Revision 1.0
*
* @brief Code for reading joystick by Craig Evans https://developer.mbed.org/users/eencae/code/Joystick/file/026fa541af7a/main.cpp
* @author Thomas Barnaby Gill
* @date   11th May 2015
*/

#include "N5110.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"
#include "WorldBuilder.h"
#include "Graphics.h"

//vcc sce rst dc  mosi clk  led
N5110 lcd(p7, p8, p9, p10, p11, p13, p26);
InterruptIn Act(p27);
InterruptIn Start(p28);
AnalogIn Noise(p19);
AnalogIn xPot(p15);
AnalogIn yPot(p16);

#define PH_MAX 20

//Joystick stuff
// timer to regularly read the joystick
Ticker pollJoystick;

// change this to alter tolerance of joystick direction
#define DIRECTION_TOLERANCE 0.05

// create enumerated type (0,1,2,3 etc. for direction)
// could be extended for diagonals etc.
enum DirectionName {
    Up,
    Down,
    Left,
    Right,
    Centre,
    Unknown
};

/** Joystick Structure
*
* It includes the x, x0, y, y0 and direction members
* @param x - current x direction potentiometer value
* @param x0 - centred x direction potentiometer value
* @param y - current y direction potentiometer value
* @param y0 - centred y direction potentiometer value
* @param direction - allowed direction values
*/
typedef struct JoyStick Joystick;
struct JoyStick {
    float x;    // current x value
    float x0;   // 'centred' x value
    float y;    // current y value
    float y0;   // 'centred' y value
    DirectionName direction;  // current direction
};
// create struct variable
Joystick joystick;

/** Calibrate Joystick
*
* Reads defualt position of joystick for further readings
*
*/
void calibrateJoystick();

/** Update Joystick
*
* Reads cuurent position of joystick relative to the defualt position
*
*/
void updateJoystick();

#define USR_POWERDOWN (0x104)

LocalFileSystem local("local"); // create local filesystem

/** Tile Structure
*
* It includes the symbol and passable members
* @param Symbol - contains the tile symbols for each tile type
* @param Passable - whether tile type can be walked on by player
*/
struct TILES {
    char    Symbol; // Symbol for this tile
    bool   Passable; // Can tile be walked on
};

TILES  TileList[] = {
    { '#', false },  // 0- WALL
    { '.', true },  // 1- FLOOR
    { '+', true }, // 2- ENTER
    { 'x', true },  // 3- EXIT
    { '.', true },  // 4- FLOOR_SEEN
    { '=', true },  // 5- CHEST
    { '/', true },  // 6- CHEST_OPENED
};

/** Item Structure
*
* It includes the ItemName and the ItemValue members
* @param ItemName - contains the name of the item as a char array
* @param ItemValue - contains damage/armour value of the item
*/
struct ITEMS {
    char  ItemName[15]; //Item name
    int ItemValue; //Damage/ armour value
};

ITEMS  ItemList[] = {
    //Weapons
    { "Dagger", 4 },  //0
    { "Axe", 5 },  //1
    { "Mace", 6 },  //2
    { "Sword", 7 },  //3
    { "Warhammer", 8 }, //4
    //Armour
    { "Cloth armour", 0 },  //5
    { "Leather armour", 1 },  //6
    { "Studded armour", 2 },  //7
    { "Chainmail vest", 3 }, //8
    { "Plate armour", 4 }, //9
};

/** Enemy Structure
*
* It includes the EName, EHealth, EDamage, EArmour, EDodge, EHit, ESpd and EPoints members
* @param EName - contains the name of the enemy as a char array
* @param EHealth - contains the monster's full health value
* @param EADamage - contains the monster's damage value
* @param EArmour - contains the monster's armour value
* @param EDodge - contains the monster's dodge chance value
* @param EHit - contains the monster's hit chance value
* @param ESpd - contains the monster's speed value
* @param EPoints-  contains the score value of the monster
*/
struct ENEMIES {
    char  EName[9]; //Enemy name
    int   EHealth; //Enemy health
    int   EDamage; //Enemy damage
    int   EArmour; //Enemy armour
    int   EDodge; //Enemy dodge chance
    int   EHit; //Enemy hit chance
    int   ESpd; //Enemy speed (used for running away during fights)
    int   EPoints; //How many points the monster is worth
};

ENEMIES  EnemyList[] = {
    //Name          HP Dmg Arm  Dg  Ht  Spd  Pts
    { "Huge Rat",   5,  3,   0,  25, 70, 40,  1  }, //0- Huge Rat
    { "Goblin",     6,  3,   1,  25, 60, 30,  3  }, //1- Goblin
    { "Skeleton",   8,  4,   2,  10, 50, 10,  5  }, //2- Skeleton
    { "Wraith",     5,  4,   0,  40, 60, 50,  10 }, //3- Wraith
    { "Ogre",       10, 7,   3,  10, 35, 15,  15 }, //4- Ogre
    { "Minotaur",   15, 8,   4,  20, 45, 100, 50 }, //5- Minotaur
};

//Variables
/** Act Flag
*
* @param ActFlag - Used in the ISR. It is set to 1 when the act button is pressed
*
*/
int ActFlag = 0;
/** Start Flag
*
* @param StartFlag - Used in the ISR. It is set to 1 when the start button is pressed
*
*/
int StartFlag = 0;

/** Player Health
*
* @param ph - contains the player's current health value
*
*/
//Player Health
int ph = PH_MAX; 

/** Player Weapon
*
* @param pw - contains the player's current weapon identifier value
*
*/
//Player weapon
int pw = 0; //0 to 4

/** Player Armour
*
* @param pa - contains the player's current armour identifier value
*
*/
//Player armour
int pa = 5; //5 to 9

/** Monster Health
*
* @param mh - contains the monster's current health value
*
*/
//Monster health
int mh;

/** Score
*
* @param score - contains the player's current score value
*
*/
//Player score
int score = -100; //100 level, x monster, 10 chest, -2 running away

/** Highest Score
*
* @param HScore1 - contains the top high score value
*/
//High scores
int HScore1;

/** Second Highest Score
*
* @param HScore2 - contains the second high score value
*
*/
int HScore2;

/** Third Highest Score
*
* @param HScore3 - contains the third high score value
*
*/
int HScore3;

/** Fourth Highest Score
*
* @param HScore4 - contains the fourth high score value
*
*/
int HScore4;

/** Boss Fight
*
* @param BossFight - Defualt false. Set to true at the exit of every level that is a multiple of 5 (labyrinth levels). Causes a minotaur to appear in the fight function.
*
*/
bool BossFight = false;

//Voids
/** Semihost Powerdown
*
* Powers down the ethernet
*
*/
int semihost_powerdown();

/** Act Button Pressed
*
* Interrupt Service Routine for the act button 
*
*/
void ActPressed();

/** Start Button Pressed
*
* Interrupt Service Routine for the act button 
*
*/
void StartPressed();

/** Write Data to File
*
* Writes the highscores to the .txt file
*
*/
void writeDataToFile();

/** Write Data to File
*
* Reads the highscores from the .txt file and stores the values obtained to the high score integers
*
*/
void readDataFromFile();

/** High Score Check
*
* Compares the current score with the saved high scores, arranges the scores in order and then saves the scores to the .txt file
*
*/
void HighScoreCheck();

/** Flash Screen
*
* Inverts and then uninverts the screen's colours
* @param n - contains the number of times the screen flashes
*
*/
void FlashScreen(int n);

/** Draw Graphic
*
* Draws a 38x38 pixel image from a starting point
* @param p - contains the which image to draw (0- Huge Rat, 1- Goblin, 2- Skeleton, 3- Wraith, 4- Ogre, 5- Minotaur)
* @param x - contains the x co-ordinate of starting position
* @param y - contains the y co-ordinate of starting position
*
*/
void DrawGraphic(int p, int x, int y);

/** Intro
*
* Displays the game title when the system is turned on
*
*/
void Intro();

/** Initilize
*
* Initilizes the score, level number and player health and equipment
* 
*
*/
void Initilize();

/** Main Menu
*
* Displays a menu where the user can choose to start the game, view highscores or view the option menu 
* 
*
*/
void MainMenu();

/** Options Menu
*
* Allows the user to turn the screen backlight on or off
* 
*
*/
void Options();

/** Level Screen
*
* Displays which level the user is about to enter
* 
*
*/
void LevelScreen();

/** Next Level
*
* Increases the score, generates the next level and then initilizes the random seed
* 
*
*/
void NextLevel();

/** Game Loop
*
* The main loop while the game is running.
* Checks if the user is interacting with a chest, moving the analogue stick or pressing a button and then calls the corresponding function
* 
*
*/
void GameLoop();

/** Player Camera
*
* Displays the map around the player as characters on the screen
*
*/
void PlayerCamera();

/** Player Move
*
* Checks which way the analogue stick was moved and then moves the player into that space if the tile is passable
*
*/
void PlayerMove();

/** Fight
*
* Function called during a fight.
* It first displays the opponent and then gives the user the option to fight or run away.
* The loop ends if the player runs away, kills the opponent or is killed.
*
*/
void Fight();

/** Monster Attack
*
* Calculates the damage the opponent does to the player
* @param m - contains the current monster identifier
*
*/
void MonsterAttack(int m);

/** Player Attack
*
* Calculates the damage the player does to the opponent
* @param m - contains the current monster identifier
*
*/
void PlayerAttack(int m);

/** Start Menu
*
* Displays a screen showing the player's current health and a menu where the user can view the map, map legend and inventory
*
*/
void StartMenu();

/** Map
*
* Calls the functions which displays the map of the current level on the screen and flashes the player's current location
*
*/
void Map();

/** Draw Map
*
* Displays the map of the current level on the screen
*
*/
void DrawMap();

/** Flash Player Location
*
* Flashes the player's current location on the screen
*
*/
void FlashPlayerLocation();

/** Map legend
*
* Displays a screen showing what each character represents on the main game screen
*
*/
void MapLegend();

/** Inventory
*
* Displays a screen showing what weapon and armour the player has equipped.
*
*/
void Inventory();

/** Chest
*
* Generates either an item, potion, map or booby trap in the chest as it is opened.
*
*/
void Chest();

/** Get Item
*
* Generates an item from the list of items and asks if the player wants to equip it.
*
*/
void getItem();

/** Booby Trap
*
* Damages the player for 0-5 damage. If the damage is 0 the trap will display as having failed. 
*
*/
void BoobyTrap();

/** Reveal Map
*
* Turns all the unseen floors into seen ones. 
*
*/
void RevealMap();

/** Potion
*
* Generates either a potion that heals the player, damages the player or transports them to the next level
*
*/
void Potion();

/** Game Over
*
* Displays the game over screen
*
*/
void GameOver();

/** Score Screen
*
* Displays the user's final score
*
*/
void ScoreScreen();

/** High Score Screen
*
* Displays the top 4 high scores
*
*/
void HighScoreScreen();