/*
 *  LICENSE
 *  Copyright (c) 2012 James Bowman, Chris Dick
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 *
 *  DESCRIPTION
 * Copyright (c) 2011 by James Bowman <jamesb@excamera.com>
 * Gameduino library for mbed. Ported from Arduino by Chris Dick
 *
 *
 */


#ifndef _GD_H_INCLUDED
#define _GD_H_INCLUDED
#include "mbed.h"
#include "arduino.h" 

struct sprplot
{
  signed char x, y;
  byte image, palette;
};

/** Gameduino library ported from the arduino
 *
 * Example:
 * @code
 * // 
 * #include "mbed.h"
 * #include "GD.h"
 * 
 * 
 * GDClass GD(p5, p6, p7, p8, USBTX, USBRX) ;
 * 
 * int main() {
 *        // initialise Gameduino
 *        GD.begin();
 *        // load ascii characters 
 *        GD.ascii();
 * }
 * @endcode
 */
class GDClass {
private:
  SPI gameduino_spi;
  DigitalOut gameduino_cs;
  Serial pc;
public:
/** Create a Gameduino Object with the specified spi pins and uart pins.
 *
 * @param mosi name of pin for spi mosi connection.
 * @param miso name of pin for spi moso connection.
 * @param sclk name of pin for spi sclk connection.
 * @param cs name of pin for spi cs connection.
 * @param utx name of pin for uart tx.
 * @param urx name of pin for uart rx.
 */
  GDClass(PinName mosi, PinName miso, PinName sclk, PinName cs, PinName utx, PinName urx);
/** Initialize the connection to the adapter, and reset the adapter.
 *
 * @param void.
 */
  void begin();
/** end the spi transaction.
 *
 * @param void.
 */
  void end();
/** start an spi transaction to addr.
 *
 * @param addr: address of memory location.
 */
  void __start(unsigned int addr);
/** start an spi write transaction to addr.
 *
 * @param addr: address of memory location. 
 */
  void __wstart(unsigned int addr);
/** end the spi transaction.
 *
 * @param void.
 */
  void __end(void);
/** Read a byte from memory location addr.
 *
 * @param addr: address of memory location.
 */
  byte rd(unsigned int addr);
/** Write byte v to memory location addr.
 *
 * @param addr: address of memory location.
 * @param v: data to write.
 */
  void wr(unsigned int addr, byte v);
/** Read a 16-bit word from memory locations addr and addr+1.
 *
 * @param addr: address of first memory location.
 */
  unsigned int rd16(unsigned int addr);
/** Write 16-bit word v to memory locations addr and addr+1.
 *
 * @param addr: address of first memory location.
 * @param v: data to write.
 */
  void wr16(unsigned int addr, unsigned int v);
/** Fill count memory locations starting at addr with value v.
 *
 * @param addr: address of first memory location.
 * @param v: data to write.
 * @param count: number of memory location to fill.
 */
  void fill(int addr, byte v, unsigned int count);
/** Copy count bytes from AVR flash memory location src to addr.
 *
 * @param addr: address of first destination memory location.
 * @param *src: address of first flash memory.
 * @param count: number of memory location to copy.
 */
  void copy(unsigned int addr, PROGMEM prog_uchar *src, int count);
/** Set character palette entry pal to color rgb.
 *
 * @param pal: address of palatte.
 * @param rgb: colour value.
 */
  void setpal(int pal, unsigned int rgb);
/** Set attributes of sprite spr, can also use RGB(r,g,b) Macro.
 *
 * @param spr: sprite number, 0-255.
 * @param x: x coordinate, 0-511. 0 is left edge of screen.
 * @param y: y coordinate, 0-511. 0 is top edge of screen.
 * @param image: image number, 0-63.
 * @param palette: palette control 0-15.
 * @param rot: rotate/flip control 0-7.
 * @param jk: J/K collision select, 0-1.
 */
  void sprite(int spr, int x, int y, byte image, byte palette, byte rot = 0, byte jk = 0);
/** Set attributes of four sprites in a 2x2 configuration. (x,y) specifies the center of the group.
 *
 * @param spr: sprite number, 0-255.
 * @param x: x center coordinate , 0-511. 0 is left edge of screen.
 * @param y: y center coordinate, 0-511. 0 is top edge of screen.
 * @param image: image number, 0-63.
 * @param palette: palette control 0-15.
 * @param rot: rotate/flip control 0-7.
 * @param jk: J/K collision select, 0-1.
 */
  void sprite2x2(int spr, int x, int y, byte image, byte palette, byte rot = 0, byte jk = 0);
/** Waits for the start of the vertical blanking period. By waiting for vertical blanking: any animation is guaranteed to be smooth, the sprite collision RAM is valid.
 *
 * @param void
 */
  void waitvblank();
/** Copy count bytes from AVR flash memory location src into the coprocessor’s instruction RAM, and restart the coprocessor.
 *
 * @param *src: address of first flash memory.
 * @param count: number of memory location to copy.
 */
  void microcode(PROGMEM prog_uchar *src, int count);
/** uncompression algorithm for use with online tools
 *
 * @param addr first address of gameduino memory to store to.
 * @param *src pointer to compressed data in program memory.
 */
  void uncompress(unsigned int addr, PROGMEM prog_uchar *src);
/** Set audio voice waveform, frequency and amplitude. The 64 audio voices are summed for final output. So to guarantee no audio clipping, the sum of all voices’ amplitudes should be 255 or less.
 *
 * @param voice: hardware voice number, 0-63.
 * @param wave: waveform, 0 is sine wave, 1 is noise.
 * @param freq: frequency in quarter-Hz, e.g. 100 Hz is 400.
 * @param lamp: left amplitude, 0-255.
 * @param ramp: right amplitude, 0-255.
 */
  void voice(int v, byte wave, unsigned int freq, byte lamp, byte ramp);
/** Load a standard 8x8 ASCII font into character RAM. This is useful for debugging and status display. 
 *
 * @param void
 *
 */
  void ascii();
/** Draw string s at character screen position (x,y). Assumes that a standard character set has been loaded, e.g. by ascii().
 *
 * @param x: x coordinate, 0-511. 0 is left edge of screen.
 * @param y: y coordinate, 0-511. 0 is top edge of screen.
 * @param *s: pointer to string.
 */
  void putstr(int x, int y, const char *s);
/** grab a screen shot and dump to uart. Python script availiable from http://excamera.com/sphinx/gameduino/samples/screenshot/index.html to save recieved data.
 *
 * @param frame number to added to saved filename.
 */
  void screenshot(unsigned int frame);
/** start an spi write transaction to sprite.
 *
 * @param spr: sprite number to write to.
 */
  void __wstartspr(unsigned int spr = 0);
/** controls sprite chosen with __wstartspr with position (x,y), offset (ox,oy), rotation rot, collision class jk and palette and image.
 *
 * @param ox: x offset.
 * @param oy: y offset.
 * @param x: x coordinate, 0-511. 0 is left edge of screen.
 * @param y: y coordinate, 0-511. 0 is top edge of screen.
 * @param image: SpriteImageSelect. 0-63. slects which source image to use.
 * @param palette: sprite palette select, 4 bits Controls how pixel data is converted to colour.
 * @param rot: rotation of sprite, 3 bits rotates sprites by quarter turns.
 * @param jk: collision class. 0 is J, 1 is K.
 */
  void xsprite(int ox, int oy, signed char x, signed char y, byte image, byte palette, byte rot = 0, byte jk = 0);
/** hide current sprite by moving it off visible screen area
 *
 * @param void
 */
  void xhide();
/** plots count sprites at using xsprite method 
 *
 * @param void
 */
  void plots(int ox, int oy, PROGMEM sprplot *psp, byte count, byte rot, byte jk);
  
  byte spr;   // Current sprite, incremented by xsprite/xhide above
};

#define GD_HAS_PLOTS    1     // have the 'GD.plots' method

//extern GDClass GD;

#define RGB(r,g,b) ((((r) >> 3) << 10) | (((g) >> 3) << 5) | ((b) >> 3))
#define TRANSPARENT (1 << 15) // transparent for chars and sprites

#define RAM_PIC       0x0000    // Screen Picture, 64 x 64 = 4096 bytes
#define RAM_CHR       0x1000    // Screen Characters, 256 x 16 = 4096 bytes
#define RAM_PAL       0x2000    // Screen Character Palette, 256 x 8 = 2048 bytes

#define IDENT         0x2800
#define REV           0x2801
#define FRAME         0x2802
#define VBLANK        0x2803
#define SCROLL_X      0x2804
#define SCROLL_Y      0x2806
#define JK_MODE       0x2808
#define J1_RESET      0x2809
#define SPR_DISABLE   0x280a
#define SPR_PAGE      0x280b
#define IOMODE        0x280c

#define BG_COLOR      0x280e
#define SAMPLE_L      0x2810
#define SAMPLE_R      0x2812

#define MODULATOR     0x2814

#define SCREENSHOT_Y  0x281e

#define PALETTE16A    0x2840   // 16-color palette RAM A, 32 bytes
#define PALETTE16B    0x2860   // 16-color palette RAM B, 32 bytes
#define PALETTE4A     0x2880   // 4-color palette RAM A, 8 bytes
#define PALETTE4B     0x2888   // 4-color palette RAM A, 8 bytes
#define COMM          0x2890   // Communication buffer
#define COLLISION     0x2900   // Collision detection RAM, 256 bytes
#define VOICES        0x2a00   // Voice controls
#define J1_CODE       0x2b00   // J1 coprocessor microcode RAM
#define SCREENSHOT    0x2c00   // screenshot line RAM
#define RAM_SPR       0x3000    // Sprite Control, 512 x 4 = 2048 bytes
#define RAM_SPRPAL    0x3800    // Sprite Palettes, 4 x 256 = 2048 bytes
#define RAM_SPRIMG    0x4000    // Sprite Image, 64 x 256 = 16384 bytes

#define GET_FAR_ADDRESS(var) (var)
#endif //_GD_H_INCLUDED