/* Bike Service
 */

#ifndef __BLE_BIKE_SERVICE_H__
#define __BLE_BIKE_SERVICE_H__

#include "ble/BLE.h"

/**
* @class BikeService
* @brief BLE Service for Heck Bike.
*/
class BikeService {
public:
    /**
     * @brief Constructor with 8-bit HRM Counter value.
     *
     * @param[ref] _ble
     *               Reference to the underlying BLE.
     */    
    BikeService(BLE &_ble) :
        ble(_ble),
        inputBytes(0),
        outputBytes(0),
        timestampBytes(),
        alarmBytes(0),
        input_characteristic(0x8EC6, inputBytes.getPointer(),
                inputBytes.getNumValueBytes(), inputBytes.getNumValueBytes(),
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        output_characteristic(0x8EC7, outputBytes.getPointer(),
                outputBytes.getNumValueBytes(), outputBytes.getNumValueBytes(),
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        timestamp_characteristic(0x8EC8, timestampBytes.getPointer(),
                timestampBytes.getNumValueBytes(), timestampBytes.getNumValueBytes(),
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        log_characteristic(0x8EC9,0),
        alarmcode_characteristic(0x8ECA,alarmBytes.getPointer(),
                alarmBytes.getNumValueBytes(), alarmBytes.getNumValueBytes(),
                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE){
        setupService();
    }
    
    /**
     * @brief Set the inputs.
     *
     * @param[in] value
     *                  The inputs, with one bit for each input.
     */
    void updateInputs(uint8_t value) {
        inputBytes.updateValue(value);
        ble.gattServer().write(input_characteristic.getValueHandle(), inputBytes.getPointer(), inputBytes.getNumValueBytes());
    }
    
    /**
     * @brief Set the outputs.
     *
     * @param[in] value
     *                  The outputs, with one bit for each output.
     */
    void updateOutputs(uint8_t value) {
        outputBytes.updateValue(value);
        ble.gattServer().write(output_characteristic.getValueHandle(), outputBytes.getPointer(), 1);
    }
    
    /**
     * @brief Set the alarm code.
     *
     * @param[in] value
     *                  The alarm code, with one byte for each output.
     */
    void updateAlarmCode(uint8_t value) {
        alarmBytes.updateAlarm(value);
        ble.gattServer().write(alarmcode_characteristic.getValueHandle(), alarmBytes.getPointer(), 1);
    }
    
    void updateTimestamp() {
        ble.gattServer().write(timestamp_characteristic.getValueHandle(), timestampBytes.getPointer(), 1);
    }

    /**
     * This callback allows the heart rate service to receive updates to the
     * controlPoint characteristic.
     *
     * @param[in] params
     *     Information about the characterisitc being updated.
     */
    virtual void onDataWritten(const GattWriteCallbackParams *params) {
        if (params->handle == output_characteristic.getValueAttribute().getHandle()) {
            updateOutputs(*(params->data));
        }
        else if (params->handle == alarmcode_characteristic.getValueAttribute().getHandle()) {
            updateAlarmCode(*(params->data));
        }
    }

protected:
    void setupService(void) {
        GattCharacteristic *charTable[] = {&input_characteristic, &output_characteristic, &timestamp_characteristic, &log_characteristic, &alarmcode_characteristic};
        GattService         bikeService(0x8EC5, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.addService(bikeService);
        ble.onDataWritten(this, &BikeService::onDataWritten);
    }

protected:
    /* Private internal representation for the bytes used to work with the value of the input and output characteristic. */
    struct ValueBytes {

        ValueBytes(uint8_t value) : valueBytes() {
            updateValue(value);
        }

        void updateValue(uint8_t value) {
            valueBytes[0] = value;
        }
        
        uint8_t       *getPointer(void) {
            return valueBytes;
        }
        
        uint8_t getValue(){
            return valueBytes[0];
        }

        const uint8_t *getPointer(void) const {
            return valueBytes;
        }

        unsigned       getNumValueBytes(void) const {
            return 1;
        }

    private:
        uint8_t valueBytes[1];
    };
    
protected:
    /* Private internal representation for the bytes used to work with the value of the input and output characteristic. */
    struct TimestampBytes {

        TimestampBytes(void): timestampBytes(){
           updateTimestamp(time(NULL));
        }
        
        TimestampBytes(time_t value) : timestampBytes() {
            updateTimestamp(value);
        }

        void updateTimestamp(time_t value) {
            timestampBytes = value;
        }
        
        uint8_t       *getPointer(void) {
            return NULL;//&timestampBytes;//TODO MAKE THIS WORK
        }

        const uint8_t *getPointer(void) const {
            return NULL;//&timestampBytes;//TODO MAKE THIS WORK
        }

        unsigned       getNumValueBytes(void) const {
            return sizeof(timestampBytes);
        }

    private:
        time_t timestampBytes;
    };
    
protected:
    /* Private internal representation for the bytes used to work with the value of the input and output characteristic. */
    struct AlarmBytes {

        AlarmBytes(void): alarmBytes(){
           //fill this in with the default alarm code 
        }
        
        AlarmBytes(uint8_t value) : alarmBytes() {
            updateAlarm(value);
        }

        void updateAlarm(uint8_t value) {
            alarmBytes[0] = value;
        }
        
        uint8_t       *getPointer(void) {
            return alarmBytes;
        }

        const uint8_t *getPointer(void) const {
            return alarmBytes;
        }

        unsigned       getNumValueBytes(void) const {
            return 5;
        }

    private:
        uint8_t alarmBytes[5];
    };
    
protected:
    BLE                 &ble;

    ValueBytes      inputBytes;
    ValueBytes      outputBytes;
    TimestampBytes  timestampBytes;
    AlarmBytes      alarmBytes;

    GattCharacteristic                   input_characteristic;
    GattCharacteristic                   output_characteristic;
    GattCharacteristic                   timestamp_characteristic;
    GattCharacteristic                   log_characteristic;
    GattCharacteristic                   alarmcode_characteristic;
};

#endif /* #ifndef __BLE_BIKE_SERVICE_H__*/