#ifndef PLAYTONE
#define PLAYTONE

#include "mbed.h"
#include "Piezo.h"

#ifndef NOTE
#define NOTE

struct Note {
    int idx;
    float value;
    };

#endif

/** Class for playing 8bit-like songs on PWM speaker.
 *  This class suports easy conversion from notes to sound from piezo speaker.
 *  
 *  Currently tested with LPC1768.\n\n
 *
 *
 *                                      Writen by: Jan Crha (TeaPack_CZ), 2016.
 *
 *  Last modified: 2016-10-02
 *
 *  Example:
 *  @code
 * #include "mbed.h"
 * 
 * #include "PlayTone.h"
 * #include "Popcorn.h"
 * 
 * PlayTone Pt(p21);
 * Popcorn Pop;
 * 
 * int main() {
 *     
 *     int PopLen = Pop.GetLen();
 *     int PopBPM = Pop.GetBPM();
 *     
 *     Note PopcornTheme[PopLen];
 *     Pop.GetPopcorn(PopcornTheme);
 *     
 *     Pt.setBPM(PopBPM);
 *     Pt.setStaccatoDuty(1/16.0);
 *     
 *     Pt.playStaccatoSequence(PopLen, PopcornTheme);
 * }
 * @endcode
 */
 
class PlayTone{

public:

    /** Constructor, chosen pin must be PWM compatible*/
    PlayTone(PinName PS);
    
    /** Function for setting BPM.
     *  \param Bpm range is from 40 to 260
     *  @returns Duration of note in ms
     */
    int setBPM(int);
    
    /** Function for transposing whole set of notes.
     */
    void transpose(int);
    
    /** Function for playing single tone.
     *  \param index of note from list
     *  \param length of note based from setted BPM
     */
    void playTone(int,float);
    
    /** Function for playing sequence of tones.
     *  \param sequence_length length of sequence to play
     *  \param notes[] list of notes indexes
     *  \param values[] list of note lengths based from setted BPM
     */
    void playSequence(int,int[],float[]);
    
    /** Function for playing sequence of tones.
     *  \param sequence_length length of sequence
     *  \param Notes[] list of Notes
     */
    void playSequence(int,Note[]);
    
    /** Function for setting constant staccato note duty cycle based on BPM
     *  and independently on note value.
     *  \param duty in range from 0 to 1
     */
    void setStaccatoDuty(float);
    
    /** Function for playing single tone in Staccato
     *  \sa playTone()
     *  \sa setStaccatoDuty()
     */
    void playStaccato(int,float);
    
    /** Function for playing sequence of tones in Staccato.
     *  \sa playSequence()
     *  \sa setStaccatoDuty()
     */
    void playStaccatoSequence(int,int[],float[]);
    
    /** Function for playing sequence of tones in Staccato.
     *  \sa playSequence()
     *  \sa setStaccatoDuty()
     */
    void playStaccatoSequence(int,Note[]);
    
    /** Function for playing rest of given length based on set BPM
     */
    void silence(float);
    
    /** Function for stopping sounds.
     */
    void stop();
    
private:
    Piezo _speaker;
    float tone[85];
    
    int offset;
    int _bpm;
    float _bpm_ms;
    
    float max_BPM;
    float min_BPM;
    
    float _duty;
};

#endif