/** Class for the Pololu1451 motor driver carrier (VNH5019)\n
 *  This class can control VNH5019 motor driver. Class supports stop command and 
 *  hold command, which uses active breaking of motor. Active position returning
 *  can not by supported without encoder. Connected pins are: INA, INB, PWM, CS.
 *  Pins ENA, ENB needs external pullups (10k) to make it work.\n\n
 *  https://www.pololu.com/product/1451\n\n
 *
 *                                      Writen by: Jan Crha (TeaPack_CZ), 2017.
 *
 *  Last modified: 2017-02-25
 *
 *  Example (LPC1768):
 *  @code
 *  #include "mbed.h"
 *  #include "MotorDriver.h"
 *  #include "math.h"
 *  
 *  Serial PC(USBTX,USBRX);
 *
 *  MotorDriver Motor(p20,p21,p22); // Connect to pins PWM, INA, INB
 *  AnalogIn senseB(p19);           //Connect to pin CS
 *  
 *  int main(){
 *  
 *  PC.baud(115200);
 *  Motor.setEnabled(true);
 *  int i = 0;
 *  
 *  while(1)
 *  {
 *      Motor.setSpeed(sin(i/180.0*3.1415));
 *      PC.printf("%.3f",senseB.read());
 *      wait(0.1);
 *  }
 *  }
 * @endcode
 */

#include "mbed.h"

class MotorDriver{
    
public:
    
    /** Class contructor.
        \param PWM output connected to PWM pin on Pololu driver
        \param DirA connect to pin INA
        \param DirB connect to pin INB
     */
    MotorDriver(PinName Pwm, PinName DirA, PinName DirB);
    
    /** Procedure for enabling/disabling motor driver
    */
    void setEnabled(bool);
    
    /** Functon returning actual state of motor driver
        @returns True/False - state of motor driver
    */
    bool isEnabled();
    
    /** Procedure for stopping movement of motor
    */
    void stop();
    
    /** Procedure for stopping motor with active break
    */
    void hold();
    
    /** Procedure for setting direction and power
        \param direction
        \param power integer < 0 ; 100 >
    */
    void setSpeed(bool, int);
    
    /** Procedure for setting power and direction via float
        \param power_f float < -1 ; +1 >
    */
    void setSpeed(float);
    
private:
    PwmOut _pwm;
    DigitalOut _dirA;
    DigitalOut _dirB;

    bool enabled;
};