#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>
#include "EasyLube.h"
#include "Lora.h"
#include "DS1820.h"

#define TEMP_DATA_PIN                       PC_1
#define TEMP_ERROR                          -1000

#define MESSAGE_PREFIX                      "HTD-LUBRICANT:"

#define LORA_NETWORK_NAME                   "MultiTech"
#define LORA_NETWORK_PASS                   "MultiTech"
#define LORA_NETWORK_FREQUENCY_SUB_BAND     1
#define LORA_MAX_ATTEMPTS                   2

 
static std::string message_prefix = MESSAGE_PREFIX;

EasyLube easyLube;
mDot* dot;

void init()
{
    // get a mDot handle
    dot = mDot::getInstance();
    
    // reset to default config so we know what state we're in
    dot->resetConfig();
    
    // Set log level to info, so everything will be sent to the output?
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);
    
    // print library version information
    logInfo("V: %s", dot->getId().c_str());
}

//Maak een bericht van de huidige activiteit met temperatuur
std::vector<uint8_t> createMessage(EasyLube::Activity activity, float temp)
{
    std::vector<uint8_t> data;
    
    //Eerst de prefix kopieren zodat het bericht aan de andere kant te herkennen is
    for (std::string::iterator it = message_prefix.begin(); it != message_prefix.end(); it++)
        data.push_back((uint8_t) *it);
    
    //Afhankelijk van de activiteit voegen we een code toe
    switch(activity)
    {
        case EasyLube::STARTING: 
            data.push_back((uint8_t) 'S'); data.push_back((uint8_t) 'T'); 
            break;
        case EasyLube::DOSAGE: 
            data.push_back((uint8_t) 'D'); data.push_back((uint8_t) 'O'); 
            break;
        case EasyLube::ERROR: 
            data.push_back((uint8_t) 'E'); data.push_back((uint8_t) 'R'); 
            break;
        case EasyLube::ALIVE: 
            data.push_back((uint8_t) 'A'); data.push_back((uint8_t) 'L'); 
            break;
        case EasyLube::NO_SIGNAL: 
            //01-09-2016 21:32 toegevoegd, werd eerst PU gestuurd
            data.push_back((uint8_t) 'N'); data.push_back((uint8_t) 'S'); 
            break;
        case EasyLube::POWER_UP:
            data.push_back((uint8_t) 'P'); data.push_back((uint8_t) 'U'); 
            break;
        default:
            data.push_back((uint8_t) 'O'); data.push_back((uint8_t) 'N'); 
    }
    
    //Als de temperatuur anders is dan de TEMP_ERROR, dan voegen we deze waarde ook toe aan het resultaat
    if(temp > TEMP_ERROR)
    {
        //Format is TEMP=123.4, met een comma vooraf om onderscheid te maken met de code
        char buffer[50];
        uint8_t n=sprintf (buffer, ",TEMP=%3.1f", temp);
        logInfo("It is %3.1foC", temp);
        for (uint8_t i = 0; i < n; i++)
        {
            data.push_back((uint8_t) buffer[i]); 
        }
    }
    return data;
}
 
int main() {
    init();
    
    //De lora instantie kan pas aangemaakt worden er een instantie is van de mDot welke in de init wordt gemaakt
    Lora lora(dot, LORA_NETWORK_NAME, LORA_NETWORK_PASS, LORA_NETWORK_FREQUENCY_SUB_BAND);
    
    //De huidige activiteit opvragen
    EasyLube::Activity activity = easyLube.getActivity();
    
    //Als er geen activiteit is dan zijn we snel klaar, is wel een bijzondere situatie
    if(activity != EasyLube::NONE)
    {
        std::vector<uint8_t> data;
        
        //Controleren of er een DS1820 aanwezig is.
        if(DS1820::unassignedProbe(TEMP_DATA_PIN))
        {
            //Zo ja, de temperatuur uitlezen
            DS1820 probe(TEMP_DATA_PIN);
            probe.convertTemperature(true);
            float temp = probe.temperature();
            if(temp != temp)
            {
                temp = TEMP_ERROR;    
            }
            //Get message with temp
            data = createMessage(activity, temp);
        }
        else
        {
            logInfo("Geen temperatuur sensor gevonden");
            //Get message without temp  
            data = createMessage(activity, TEMP_ERROR);  
        }
        
        //De gegevens versturen
        lora.SendData(data, LORA_MAX_ATTEMPTS);
    }

    //32 minuten slapen of wachten op een interrupt
    dot->sleep(1920, mDot::RTC_ALARM_OR_INTERRUPT, true);

    return 0;
}
