
#include "Lora.h"


Lora::Lora(mDot* dot, std::string networkName, std::string networkPassword, uint8_t frequencySubBand)
{
    _dot = dot;
    _networkName = networkName;
    _networkPassword = networkPassword;  
    _frequencySubBand = frequencySubBand; 
    _isInitialized = 0;   
}

void Lora::config()
{
    if(_isInitialized)
    {
        logInfo("Lora is already initialized");
        return;    
    }
    
    logInfo("Initialize Lora");
    
    int32_t ret;
    Timer configTimer;
    configTimer.reset();
    configTimer.start();

    // frequency sub band is only applicable in the 915 (US) frequency band
    // if using a MultiTech Conduit gateway, use the same sub band as your Conduit (1-8) - the mDot will use the 8 channels in that sub band
    // if using a gateway that supports all 64 channels, use sub band 0 - the mDot will use all 64 channels
    logInfo("setting frequency sub band");
    if ((ret = _dot->setFrequencySubBand(_frequencySubBand)) != mDot::MDOT_OK) {
        logError("failed to set frequency sub band %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    logInfo("setting network name");
    if ((ret = _dot->setNetworkName(_networkName)) != mDot::MDOT_OK) {
        logError("failed to set network name %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    logInfo("setting network password");
    if ((ret = _dot->setNetworkPassphrase(_networkPassword)) != mDot::MDOT_OK) {
        logError("failed to set network password %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    logInfo("setting TX spreading factor");
    if ((ret = _dot->setTxDataRate(mDot::SF_10)) != mDot::MDOT_OK) {
        logError("failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // request receive confirmation of packets from the gateway
    logInfo("enabling ACKs");
    if ((ret = _dot->setAck(1)) != mDot::MDOT_OK) {
        logError("failed to enable ACKs %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // set join mode to AUTO_OTA so the mDot doesn't have to rejoin after sleeping
    logInfo("setting join mode to AUTO_OTA");
    if ((ret = _dot->setJoinMode(mDot::AUTO_OTA)) != mDot::MDOT_OK) {
        logError("failed to set join mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("saving config");
    if (! _dot->saveConfig()) {
        logError("failed to save configuration");
    }
    //*******************************************
    // end of configuration
    //*******************************************

    logInfo("Configuratie duurde: %f seconds", configTimer.read()); 
    configTimer.stop();
    
    _isInitialized = 1;
}

void Lora::SendData(std::vector<uint8_t> data, uint8_t maxAttempts)
{
    int32_t ret;
    int attempts = 0;
    
    config();
    
    printf("[INFO] Message: ");
    for(std::vector<uint8_t>::iterator it = data.begin(); it != data.end(); ++it) {
        printf("%c", (char)*it);
    }
    printf("\r\n");
    
    logInfo("joining network");
    while ((ret = _dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        //osDelay(std::max((uint32_t)1000, (uint32_t)dot->getNextTxMs()));
        
        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        uint32_t sleep_time = std::max((uint32_t)10000, (uint32_t)_dot->getNextTxMs()) / 1000;
        
        attempts++;
        if(attempts >= maxAttempts)
        {
            logError("Versturen niet gelukt.... afbreken");
            return;    
        }
        
        logInfo("going to sleep...");

        osDelay(std::max((uint32_t)5000, (uint32_t)_dot->getNextTxMs()));
    }

    attempts = 0;

    while (true) {
        // send the data to the gateway
        if ((ret = _dot->send(data)) != mDot::MDOT_OK) {
            logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("successfully sent data to gateway");
            return;
        }

        attempts++;
        if(attempts >= maxAttempts)
        {
            logError("Versturen niet gelukt.... afbreken");
            return;    
        }
        
        logInfo("going to sleep...");
        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max((uint32_t)5000, (uint32_t)_dot->getNextTxMs()));
    }
}