#ifndef INCLUDED_I2C_h_
#define INCLUDED_I2C_h_

#include "mbed.h"
/* マクロ定義 ----------------------------------------------------------------*/

/**
    @brief  MDデータ構造内のモーターに出力するPWM値をセットする
    @param  __MD_DATA__: MDデータ構造
    @param  __NUMBER__: MDデータ構造の番号
    @param  __PWMVAL__: セットするPWMの値
    @retval None
*/
#define MD_SET_PWM(__MD_DATA__, __NUMBER__, __PWMVAL__) \
  (((__MD_DATA__) + (__NUMBER__))->PWMVal = (__PWMVAL__))

/**
    @brief  MD用データ構造内のモーターの動作状態をセットする
    @param  __MD_DATA__: MDデータ構造
    @param  __NUMBER__: MDデータ構造の番号
    @param  __STATUS__: 動作させたいモーターの状態
        @arg MD_BRAKE   ブレーキ
        @arg MD_FORWARD 正転
        @arg MD_REVERSE 逆転
    @retval None
*/
#define MD_SET_DRIVE(__MD_DATA__, __NUMBER__, __STATUS__) \
  (((__MD_DATA__) + (__NUMBER__))->DriveStatus = (__STATUS__))

/**
    @brief  MD用にMDデータ構造にセットされたPWM値を取得する
    @param  __MD_DATA__: MDデータ構造
    @param  __NUMBER__: MDデータ構造の番号
    @retval PWM値
*/
#define MD_GET_PWM(__MD_DATA__, __NUMBER__) \
  (((__MD_DATA__) + (__NUMBER__))->PWMVal)

/**
    @brief  MD用にMDデータ構造にセットされたモーターの動作状態を取得する
    @param  __MD_DATA__: MDデータ構造
    @param  __NUMBER__: MDデータ構造の番号
    @retval モーターの動作状態
*/
#define MD_GET_DRIVE(__MD_DATA__, __NUMBER__) \
  (((__MD_DATA__) + (__NUMBER__))->DriveStatus)
  
  
  
#define GET_IN(__IN_I2C_Data__,__NUM__) (__IN_I2C_Data__+(__NUM__)) -> in_data
#define GET_ADC(__ADC_I2C_Data__,__NUM__) (__ADC_I2C_Data__+(__NUM__)) ->adc_data
#define GET_ENC(__ENC_I2C_Data__,__NUM__) (__ENC_I2C_Data__ + (__NUM__))->enc_data


typedef struct {
    uint8_t             DevAddress;     /* 通信相手デバイスのアドレス */
    uint8_t             Number;         /* PICから出力出来る2つのMDの出力先指定(0or1) */
    uint8_t             PWMVal;         /* モーターに出力するPWM値(0～100) */
    uint8_t             DriveStatus;    /* モーターの動作状態MD_DriveStatus_TypeDefを代入 */
    char                pData[2];       /* PICに送信するデータ用領域 */
} MD_I2C_Data_TypeDef;

typedef struct {
    uint8_t DevAddress;
    uint8_t Number;
    bool    out_data;
} OUT_I2C_Data_TypeDef;

typedef struct {
    uint8_t DevAddress;
    uint8_t Number;
    bool    in_data;
} IN_I2C_Data_TypeDef;

typedef struct {
    uint8_t DevAddress;
    uint8_t Number;
    uint16_t adc_data;
} ADC_I2C_Data_TypeDef;

typedef struct {
    uint8_t DevAddress;
    int32_t enc_data;
} ENC_I2C_Data_TypeDef;

typedef struct {
    uint8_t DevAddress;
    uint8_t Number;
    uint16_t pulse;
} SERVO_I2C_Data_TypeDef;

typedef enum {
    MD_BRAKE       = 0x00U,
    MD_FORWARD     = 0x01U,
    MD_REVERSE     = 0x02U
} MD_DriveStatus_TypeDef;

class Y_I2C
{
private:
    I2C i2c;

public:
    Y_I2C(PinName sda, PinName scl);
    void Out_Set(OUT_I2C_Data_TypeDef *OUT_I2C_Data, uint8_t num, uint8_t value);
    void Out(OUT_I2C_Data_TypeDef *OUT_I2C_Data, uint8_t MAX);
    void Servo_Set(SERVO_I2C_Data_TypeDef *SERVO_I2C_Data, uint8_t num, uint16_t pulse);
    void Servo(SERVO_I2C_Data_TypeDef *SERVO_I2C_Data, uint8_t MAX);
    void In(IN_I2C_Data_TypeDef *IN_I2C_Data, uint8_t num);
    void Adc(ADC_I2C_Data_TypeDef *ADC_I2C_Data, uint8_t num);
    void Enc(ENC_I2C_Data_TypeDef *ENC_I2C_Data, uint8_t num);
    void MD_I2C(MD_I2C_Data_TypeDef *MD_I2C_Data, uint8_t num);
    void MD_I2C_Empty(MD_I2C_Data_TypeDef *MD_I2C_Data, uint8_t num);
};

#endif