#ifndef _S11059_H_
#define _S11059_H_
#include "mbed.h"
/**
 * S11059 HAMAMATSU
 * @note RGBUr Color Sensor with I2C Interface
 * @note I2C 7bit address: 0x2A
 *
 */

class S11059
{
public:
 /**
 *  constructor
 *
 * @param sda SDA pin
 * @param scl SCL pin
 * @param addr address of the I2C peripheral
 */
 S11059(PinName sda, PinName scl, int addr) ;

 ~S11059() ;

 /*
  * some member functions here (yet to be written)
  */
/**
 * Get measured Red Value as float [0..1]
 * @returns the value of Red as float
 */
float    getR(void) ; // return float value of Red

/**
 * Get measured Green Value as float [0..1]
 * @returns the value of Green as float
 */
float    getG(void) ; // return float value of Green

/**
 * Get measured Blue Value as float [0..1]
 * @returns the value of Blue as float
 */
float    getB(void) ; // return float value of Blue

/**
 * Get measured Infrared Value as float [0..1]
 * @returns the value of Infrared as float
 */
float    getIR(void) ; // return float value of Infrared

/**
 * Get measured value of Red
 * @param rdata unsigned 16bit data of Red
 */
void     getRData(uint16_t *rdata) ;

/**
 * Get measured value of Green
 * @param rdata unsigned 16bit data of Green
 */
void     getGData(uint16_t *gdata) ;

/**
 * Get measured value of Blue
 * @param rdata unsigned 16bit data of Blue
 */
void     getBData(uint16_t *bdata) ;

/**
 * Get measured value of Infrared
 * @param rdata unsigned 16bit data of Infrared
 */
void     getIRData(uint16_t *irdata) ;

/**
 * Get measured values of Red, Green, Blue and Infrared at once
 * @param 4 unsigned 16bit data data[0]:Red, data[1]:Green, data[2]:Blue, data[3]:Infrared
 */
void     getAllData(uint16_t *data) ; /* pass uint16_t data[4] */

 /**
 * Get value of the Control Register
 * @returns the value of Control Register at address 0x00
 * @note bit[7] ADC Reset 1:reset 0:start
 * @note bit[6] Sleep function 1:sleep 0:active
 * @note bit[5] Sleep monitor
 * @note bit[4] (reserved)
 * @note bit[3] Gain 1:High Gain 0:Low Gain
 * @note bit[2] Integration mode 1:manual 0:fixed
 * @note bit[1:0] Integration time
 * @note (00)87.5us (01)1.4ms (10)22.4ms (11)179.2ms
 *
 */
uint8_t  getControl(void) ;

/**
 * Set value to the Control Register
 * @param ctrlValue value to assing to the Control Register at address 0x00
 */
void     setControl(uint8_t ctrlValue) ;

/**
 * Get value of the Manual Timing Register at address 0x01-0x02
 * @returns 16bit value of Manual Timing Register
 * @note This value matters only in Manual Mode (bit[2] == 1)
 * @note In manual mode the integration time is
 * @note length specified by bit[1:0] of Control Register (address 0x00)
 * @note multiplied by the value of address 0x01-0x02
 * @note if the value of 0x01-0x02 is N
 * @note and according to the bit[1:0] of Control Register
 * @note Integration time length is calculated as
 * @note (00) 175   * N us
 * @note (01)   2.8 * N ms
 * @note (10)  44.8 * N ms
 * @note (11) 358.4 * N ms
 */
uint16_t getTiming(void) ;

/**
 * Set value to the Manual Timing Register at address 0x01-0x02
 * @param timingValue 16bit timing value MSB to 0x01 LSB to 0x02
 */
void     setTiming(uint16_t timingValue) ;

/**
 * Set/clear ADC Reset bit in the Control Register
 * @param mode 1:reset 0:start
 */
void     setADCReset(int mode) ;

/**
 * Get value of the ADC Reset bit (bit[7]) in the Control Register
 * @returns the ADC Reset bit (bit[7])
 */
int      getADCReset(void) ;

/**
 * Set value to the sleep function bit (bit[6]) in the Control Register
 * @param mode bit value 1:sleep 0:active
 */
void     setSleepMode(int mode) ;

/**
 * Get value of the sleep function bit (bit[6]) in the Control Register
 * @returns bit value of sleep function bit (bit[6])
 */
int      getSleepMode(void) ;

/**
 * Set value to the gain bit (bit[3]) in the Control Register
 * @param mode bit value to assign 1:High Gain 0:Low Gain
 */
void     setGain(int mode) ;

/**
 * Get value of the gain bit (bit[3]) in the Control Register
 * @returns the bit value of gain bit (bit[3])
 */
int      getGain(void) ;

/**
 * Set value to Integration mode bit (bit[2]) of the Control Register
 * @param mode bit value to assign 1:Manual Mode 0:Fixed Time Mode
 */
void     setIntegralMode(int mode) ;

/**
 * Get value of Integration mode bit (bit[2]) of the Control Register
 * @returns the bit value of Integration mode bit (bit[2])
 */
int      getIntegralMode(void) ;

private:
  I2C m_i2c;
  int m_addr;
  void readRegs(int addr, uint8_t * data, int len);
  void writeRegs(uint8_t * data, int len);
} ;

#endif /* _S11059_H_ */
