/*
 * HAMAMATSU Color Senosr S11059
 *
 */
 #include "mbed.h"
 #include "S11059.h"

 /* S11059 Register Address */
 #define REG_CONTROL       0x00
 #define REG_TIMING_DATA   0x01
 #define REG_TIMING_MSB    0x01
 #define REG_TIMING_LSB    0x02
 #define REG_R_DATA        0x03
 #define REG_R_MSB         0x03
 #define REG_R_LSB         0x04
 #define REG_G_DATA        0x05
 #define REG_G_MSB         0x05
 #define REG_G_LSB         0x06
 #define REG_B_DATA        0x07
 #define REG_B_MSB         0x07
 #define REG_B_LSB         0x08
 #define REG_IR_DATA       0x09
 #define REG_IR_MSB        0x09
 #define REG_IR_LSB        0x0A

 /* register bits */
 /* REG_CONTROL (00) */
 // bit[7] ADC Reset 0:normal 1:reset
 #define BIT_ADC_RESET     0x80
 // bit[6] Sleep function 0:normal 1:standby
 #define BIT_SLEEP_MODE    0x40
 // bit[5] Sleeep function monitor bit
 #define BIT_SLEEP_MONITOR 0x20
 // bit[4] (reserved)
 // bit[3] Gain select 0:Low Gain 1:High Gain
 #define BIT_HIGH_GAIN     0x08
 // bit[2] Ingegration mode 0:Fixed 1:Manual
 #define BIT_INT_MANUAL    0x02
 // bit[1:0] Integration Time
 #define BIT_INT_LEN0      0x00 /*  87.5 us */
 #define BIT_INT_LEN1      0x01 /*   1.4 ms */
 #define BIT_INT_LEN2      0x02 /*  22.4 ms */
 #define BIT_INT_LEN3      0x03 /* 179.2 ms */

/* constructor and destructor */

 S11059::S11059(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr<<1) {
    // activate the peripheral
}

S11059::~S11059() { }

float S11059::getR(void)  // return float value of Red
{
    uint16_t r_data ;
    float data ;
    getRData(&r_data) ;
    data = (float)r_data / 65535.0 ;
    return(data) ;
}

float S11059::getG(void)  // return float value of Green
{
    uint16_t g_data ;
    float data ;
    getGData(&g_data) ;
    data = (float)g_data / 65535.0 ;
    return(data) ;
}

float S11059::getB(void)  // return float value of Blue
{
    uint16_t b_data ;
    float data ;
    getBData(&b_data) ;
    data = (float)b_data / 65535.0 ;
    return(data) ;
}

float S11059::getIR(void)  // return float value of Infrared
{
    uint16_t ir_data ;
    float data ;
    getIRData(&ir_data) ;
    data = (float)ir_data / 65535.0 ;
    return(data) ;
}

void S11059::getRData(uint16_t *rdata)
{
    uint8_t data[2] ;
    readRegs(REG_R_DATA, data, 2) ;
    *rdata = (data[0]<<8) | data[1] ;
}

void S11059::getGData(uint16_t *gdata)
{
    uint8_t data[2] ;
    readRegs(REG_G_DATA, data, 2) ;
    *gdata = (data[0]<<8) | data[1] ;
}

void S11059::getBData(uint16_t *bdata)
{
    uint8_t data[2] ;
    readRegs(REG_B_DATA, data, 2) ;
    *bdata = (data[0]<<8) | data[1] ;
}

void S11059::getIRData(uint16_t *irdata)
{
    uint8_t data[2] ;
    readRegs(REG_IR_DATA, data, 2) ;
    *irdata = (data[0]<<8) | data[1] ;
}

void S11059::getAllData(uint16_t *data)
{
    uint8_t raw_data[8] ;
    readRegs(REG_R_DATA, raw_data, 8) ;
    data[0] = (raw_data[0] << 8) | raw_data[1] ;
    data[1] = (raw_data[2] << 8) | raw_data[3] ;
    data[2] = (raw_data[4] << 8) | raw_data[5] ;
    data[3] = (raw_data[6] << 8) | raw_data[7] ;
}

uint8_t S11059::getControl(void)
{
    uint8_t data[1] ;
    readRegs(REG_CONTROL, data, 1) ;
    return(data[0]) ;
}

void S11059::setControl(uint8_t ctrlValue)
{
    uint8_t data[2] ;
    data[0] = REG_CONTROL ;
    data[1] = ctrlValue ;
    writeRegs(data, 2) ;
}

uint16_t S11059::getTiming(void)
{
    uint8_t data[2] ;
    uint16_t timing ;
    readRegs(REG_TIMING_DATA, data, 2) ;
    timing = (data[0] << 8) | data[1] ;
    return(timing) ;
}

void S11059::setTiming(uint16_t timingValue)
{
    uint8_t data[3] ;
    data[0] = REG_TIMING_DATA ;
    data[1] = (timingValue >> 8) & 0xFF ;
    data[2] = timingValue & 0xFF ;
    writeRegs(data, 3) ;
}

void S11059::setADCReset(int mode)
{
    uint8_t data[2] ;
    data[0] = REG_CONTROL ;
    readRegs(data[0], &data[1], 1) ;
    if (mode == 0) {
        data[1] ^= BIT_ADC_RESET  ;
    } else {
        data[1] |= BIT_ADC_RESET ; /* 0x80 */
    }
    writeRegs(data, 2) ;
}

int  S11059::getADCReset(void)
{
    uint8_t data[1] ;
    int result = 0 ;
    readRegs(REG_CONTROL, data, 1) ;
    if (data[0] & BIT_ADC_RESET) {
        result = 1 ;
    }
    return(result) ;
}

void S11059::setSleepMode(int mode)
{
    uint8_t data[2] ;
    data[0] = REG_CONTROL ;
    readRegs(data[0], &data[1], 1) ;
    if (mode == 0) {
        data[1] ^= BIT_SLEEP_MODE ;
    } else {
        data[1] |= BIT_SLEEP_MODE ;
    }
    writeRegs(data, 2) ;
}

int  S11059::getSleepMode(void)
{
    uint8_t data[1] ;
    int result = 0 ;
    readRegs(REG_CONTROL, data, 1) ;
    if (data[0] & BIT_SLEEP_MODE) {
        result = 1 ;
    }
    return(result) ;
}

void S11059::setGain(int mode)
{
    uint8_t data[2] ;
    data[0] = REG_CONTROL ;
    readRegs(data[0], &data[1], 1) ;
    if (mode == 0) {
        data[1] ^= BIT_HIGH_GAIN ;
    } else {
        data[1] |= BIT_HIGH_GAIN ;
    }
}

int  S11059::getGain(void)
{
    uint8_t data[1] ;
    int result = 0 ;
    readRegs(REG_CONTROL, data, 1) ;
    if (data[0] & BIT_HIGH_GAIN) {
        result = 1 ;
    }
    return(result) ;
}

void S11059::setIntegralMode(int mode)
{
    uint8_t data[2] ;
    data[0] = REG_CONTROL ;
    readRegs(data[0], &data[1], 1) ;
    if (mode == 0) {
        data[1] ^= BIT_INT_MANUAL ;
    } else {
        data[1] |= BIT_INT_MANUAL ;
    }
}

int  S11059::getIntegralMode(void)
{
    uint8_t data[1] ;
    int result = 0 ;
    readRegs(REG_CONTROL, data, 1) ;
    if (data[0] & BIT_INT_MANUAL) {
        result = 1 ;
    }
    return(result) ;
}

void S11059::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    m_i2c.read(m_addr, (char *)data, len);
}

void S11059::writeRegs(uint8_t * data, int len) {
    m_i2c.write(m_addr, (char *)data, len);
}
