//------------------------------------------------------------
//Include Header Files
//------------------------------------------------------------ 
#include "mbed.h"
#include "ble/BLE.h"


//------------------------------------------------------------
//Definition
//------------------------------------------------------------ 
#define TXRX_BUF_LEN 20                     //max 20[byte]
#define DEVICE_LOCAL_NAME "NampassCar"     
#define ADVERTISING_INTERVAL 160            //160 * 0.625[ms] = 100[ms]
#define TICKER_TIME 200000                  //200000[us] = 200[ms]
#define DIGITAL_OUT_PIN P0_9
#define ANALOG_IN_PIN   P0_4


//------------------------------------------------------------
//Command Code
//------------------------------------------------------------
uint8_t ForwardComm[4] = {0, 0, 0, 0};
uint8_t BackComm[4] = {1, 1, 1, 1};


//------------------------------------------------------------
//Object generation
//------------------------------------------------------------ 
BLE ble;
DigitalOut ForwardPin(P0_6);
DigitalOut BackPin(P0_7); 
AnalogIn ANALOG(ANALOG_IN_PIN);


//------------------------------------------------------------
//Service & Characteristic Setting
//------------------------------------------------------------ 
//Service UUID
static const uint8_t base_uuid[] = { 0x71, 0x3D, 0x00, 0x00, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;

//Characteristic UUID
static const uint8_t tx_uuid[]   = { 0x71, 0x3D, 0x00, 0x03, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;
static const uint8_t rx_uuid[]   = { 0x71, 0x3D, 0x00, 0x02, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;

//Characteristic Value
uint8_t txPayload[TXRX_BUF_LEN] = {0,};
uint8_t rxPayload[TXRX_BUF_LEN] = {0,};

//Characteristic Property Setting etc
GattCharacteristic  txCharacteristic (tx_uuid, txPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic  rxCharacteristic (rx_uuid, rxPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY| GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic *myChars[] = {&txCharacteristic, &rxCharacteristic};

//Service Setting
GattService         myService(base_uuid, myChars, sizeof(myChars) / sizeof(GattCharacteristic *));


//======================================================================
//onDisconnection
//======================================================================
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    ble.startAdvertising();
}


bool checkBuffMatch(uint8_t *buf1, uint8_t *buf2, uint8_t length){
    for(int i = 0; i < length; i++){
        if(*(buf1 + i) != *(buf2 + i)) return false;
    }
    
    return true;
}

//======================================================================
//onDataWritten
//======================================================================
void WrittenHandler(const GattWriteCallbackParams *Handler)
{   
    uint8_t buf[4] = {0};
    uint16_t bytesRead;
    
    if (Handler->handle == txCharacteristic.getValueAttribute().getHandle()) 
    {
        ble.readCharacteristicValue(txCharacteristic.getValueAttribute().getHandle(), buf, &bytesRead);
        memset(txPayload, 0, TXRX_BUF_LEN);
        memcpy(txPayload, buf, TXRX_BUF_LEN); 
       
        if(checkBuffMatch(buf, ForwardComm, 4)){
            ForwardPin = 1;
            BackPin = 0;
            wait(1);
            BackPin = 0;
            ForwardPin = 0;
        }else if(checkBuffMatch(buf, BackComm, 4)){
            ForwardPin = 0;
            BackPin = 1;
            wait(1);
            BackPin = 0;
            ForwardPin = 0;
        }else{
            printf("failed\r\n");
        }
    }
}


//======================================================================
//onTimeout
//======================================================================
void m_status_check_handle(void)
{   
    uint8_t buf[2];

    //Read Analog port
    float s = ANALOG;
    uint16_t value = s*1024; 
    buf[0] = (value >> 8);
    buf[1] = value;
    
    //Send out
    ble.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), buf, 2); 
}


//======================================================================
//convert reverse UUID
//======================================================================
void reverseUUID(const uint8_t* src, uint8_t* dst)
{
    int i;
    
    for(i=0;i<16;i++)
        dst[i] = src[15 - i];
}


//======================================================================
//main
//======================================================================
int main(void)
{
    uint8_t base_uuid_rev[16];

    //Timer Setting [us]
    Ticker ticker;
    ticker.attach_us(m_status_check_handle, TICKER_TIME);
    
    //BLE init
    ble.init();
    
    //EventListener
    ble.onDisconnection(disconnectionCallback);
    ble.onDataWritten(WrittenHandler);  

    //------------------------------------------------------------
    //setup advertising 
    //------------------------------------------------------------
    //Classic BT not support
    ble.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    
    //Connectable to Central
    ble.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    
    //Local Name
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME,
                                    (const uint8_t *)DEVICE_LOCAL_NAME, sizeof(DEVICE_LOCAL_NAME) - 1);
    
    //GAP AdvertisingData                                
    reverseUUID(base_uuid, base_uuid_rev);  
    ble.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                    (uint8_t *)base_uuid_rev, sizeof(base_uuid));
                                    
    //Advertising Interval 
    ble.setAdvertisingInterval(ADVERTISING_INTERVAL);

    //Add Service
    ble.addService(myService);
    
    //Start Advertising
    ble.startAdvertising(); 
    
    //------------------------------------------------------------
    //Loop
    //------------------------------------------------------------
    while(1)
    {
        ble.waitForEvent(); 
    }
}