#include "Mseq.h"

/**
 * @brief コンストラクタ
 * @param uint8_t n M系列信号を発生させるレジスタの個数
 */
Mseq::Mseq(uint8_t _n)
{
    reset(_n);
}

/**
 * @brief レジスタの個数を変更しつつ初期化する
 * @param uint8_t n M系列信号を発生させるレジスタの個数
 * @details n bitシフトレジスタから生成されるM系列の一周期の長さは2^n-1
            レジスタ(reg)の初期化と、nに応じたフィードバック係数を設定する 
 */
void Mseq::reset(uint8_t _n)
{
    n = _n;
    
    // フィードバック係数を更新
    // フィードバック係数は以下の資料から引用
    // https://www.xilinx.com/support/documentation/application_notes/xapp052.pdf
    switch(n)
    {
        case 3:
            coef = 5;
            break;
        case 4:
            coef = 9;
            break;
        case 5:
            coef = 9;
            break;
        case 6:
            coef = 33;
            break;
        case 7:
            coef = 65;
            break;
        case 8:
            coef = 113;
            break;
        case 9:
            coef = 33;
            break;
        case 10:
            coef = 129;
            break;
        case 11:
            coef = 513;
            break;
        case 12:
            coef = 83;
            break;
        case 13:
            coef = 27;
            break;
        case 14:
            coef = 43;
            break;
        case 15:
            coef = 16385;
            break;
        case 16:
            coef = 40977;
            break;
        
    }
    reset();
}


/**
 * @brief レジスタのみを初期化する
 * @param なし
 * @details 常にレジスタの出力に近い側を1に初期化する 
 */
void Mseq::reset()
{
    srand((unsigned int)time(NULL));
    reg = (rand() % (n-1)) + 1;
}


/**
 * @brief M系列信号を1step更新し、出力を得る
 * @param なし
 * @details
 *  p1, p2, ... pk \in aの時、regは以下の2ステップで更新される
 *  step1 : reg[j] = reg[j+1] \forall j \in {0,2,...n-1}
 *  step2 : reg[n] = reg[p1] xor reg[p2] xor .... reg[pk]
 */
uint8_t Mseq::update()
{
    uint32_t feedback = 0;
    uint32_t t = reg & coef;
    for(int i = 0; i < n; i++)
    {
        feedback ^= t & 1;
        t >>= 1;
    }
    feedback &= 1;
    reg = reg >> 1;
    reg = ((~(0xffffffff << (n-1))) & reg) | (feedback << (n -1));
    return getOutput();
}

/**
 * @brief 出力を得る
 * @param なし
 * @details 出力 (0 or 1) 
 */
uint8_t Mseq::getOutput() const
{
    return reg & 1;
}

/**
 * @brief シフトレレジスタ全体の状態を得る
 * @param なし
 * @details シフトレジスタの生データ 
 */
uint32_t Mseq::getRegister() const
{
    return reg;
}