#ifndef AMT102_H
#define AMT102_H

#include "mbed.h"
#include "Fpointer.h"
#include "math.h"

#ifndef M_PI
#define M_PI 3.14159265f
#endif

class AMT102
{
public:
   typedef enum EMODE
   {
      POLLING = 0,
      IRQ = 1,
      IRQ_NO_JAMMING = 2,
      ANG_VEL = 4,
   }EMODE;

   typedef enum Angle_Unit
   {
       DEGREE,
       RADIAN
   }Angle_Unit;

   typedef enum PPR
   {
      A = 48,
      B = 96

   }PPR;

   typedef enum AngularVelocity_Unit
   {
      RPM,
      RPS,
      RPMS,
      RPUS
   }AngularVelocity_Unit;

   AMT102(PinName _A, PinName _B, PPR ppr, PinName _I, EMODE eMode, AngularVelocity_Unit _angv_unit = RPM, Angle_Unit _ang_unit = RADIAN);

   ~AMT102();

   int read()
   {
      return _counter;
   }

   operator int()
   {
      return _counter;
   }

   void write(int counter)
   {
       _counter = counter;
   }

   AMT102& operator= (int counter)
   {
       write(counter);
       return *this;
   }

   void poll()
   {
       ProcessISR();
   }

   void reset();

   void setIndexTrigger(bool bIndexTrigger)
   {
       _bIndexTrigger = bIndexTrigger;
   }

   void attachCounterChange(void (*function)(int) = 0)
   {
       fPointerCounterChange.attach(function);
   }

   template<class T>
   void attachCounterChange(T* obj, void (T::*method)(int) = 0)
   {
       fPointerCounterChange.attach(obj, method);
   }

   void attachDirectionChange(void (*function)(int) = 0)
   {
       fPointerDirectionChange.attach(function);
   }

   template<class T>
   void attachDirectionChange(T* obj, void (T::*method)(int) = 0)
   {
       fPointerDirectionChange.attach(obj, method);
   }

   void attachIndexTrigger(void (*function)(int) = 0)
   {
       fPointerIndexTrigger.attach(function);
   }

   template<class T>
   void attachIndexTrigger(T* obj, void (T::*method)(int) = 0)
   {
       fPointerIndexTrigger.attach(obj, method);
   }


    float getSpeedFactor() {
        switch (angv_unit)
        {
            case RPM:
                return 1.0f/(float)(_ppr*4.0f*60.0f);

            case RPS:
                return 1.0f/(float)(_ppr*4.0f);

            case RPMS:
                return 1.0f/(float)(_ppr*4.0f/60.0f);

            case RPUS:
                return 1.0f/(float)(_ppr*4.0f/3600.0f);

            default:
                break;
        }
    }

    float getPositionFactor () {
        switch (ang_unit)
        {
            case DEGREE:
                return 360.0f;

            case RADIAN:
                return (float)(2.0f*M_PI);
        }
    }

   float getAngle();
   float getAngularVelocity();
   void set_Angle_Unit(Angle_Unit _unit);
   void set_AngularVelocity_Unit(AngularVelocity_Unit _unit);

protected:
   InterruptIn _pinA, _pinB;
   DigitalIn _pinI;
   Fpointer fPointerCounterChange;
   Fpointer fPointerDirectionChange;
   Fpointer fPointerIndexTrigger;

   int _counter;
   short _state;
   short _eMode;
   bool _bIndexTrigger;

   Timer _SpeedTimer;

   unsigned int _nSpeedLastTimer;
   unsigned int _nSpeedTimeoutMax;
   unsigned int _nSpeedTimeoutCount;
   int _nSpeedAvrTimeSum;
   int _nSpeedAvrTimeCount;
   float _fLastSpeed;
   PPR _ppr;
   float _angle;
   float _angularVelocity;

   Angle_Unit ang_unit;
   AngularVelocity_Unit angv_unit;

   void ProcessISR(void);
   void callback_timeout();

private:
   static short _modeLUT[32];
};

#endif