#include "AMT102.h"

#define STATE 0xC
#define MASK  0x1C
#define INC   0x40
#define DEC   0x80
#define CHG   0xC0
#define DIR   0x20
#define A     1
#define AB    3
#define B     2
#define S0    0x0
#define S1    0x4
#define S2    0x8
#define S3    0xC
#define CCW   0
#define CW    0x10

short AMT102::_modeLUT[32] =
{
   // act state S0 in CCW direction
   CCW | S0,
   CW  | S1 | A  | INC | DIR,
   CCW | S0 | B,
   CCW | S3 | AB | DEC,
   // act state S1 in CCW direction
   CCW | S1,
   CCW | S1 | A,
   CCW | S0 | B  | DEC,
   CW  | S2 | AB | INC | DIR,
   // act state S2 in CCW direction
   CCW | S1      | DEC,
   CCW | S2 | A,
   CW  | S3 | B  | INC | DIR,
   CCW | S2 | AB,
   // act state S3 in CCW direction
   CW  | S0      | INC | DIR,
   CCW | S2 | A  | DEC,
   CCW | S3 | B,
   CCW | S3 | AB,

   // act state S0 in CW direction
   CW  | S0,
   CW  | S1 | A  | INC,
   CW  | S0 | B,
   CCW | S3 | AB | DEC | DIR,
   // act state S1 in CW direction
   CW  | S1,
   CW  | S1 | A,
   CCW | S0 | B  | DEC | DIR,
   CW  | S2 | AB | INC,
   // act state S2 in CW direction
   CCW | S1      | DEC | DIR,
   CW  | S2 | A,
   CW  | S3 | B  | INC,
   CW  | S2 | AB,
   // act state S3 in CW direction
   CW  | S0      | INC,
   CCW | S2 | A  | DEC | DIR,
   CW  | S3 | B,
   CW  | S3 | AB
};

AMT102::AMT102(PinName _A, PinName _B, PPR ppr, PinName _I, EMODE eMode, AngularVelocity_Unit _angv_unit, Angle_Unit _ang_unit)
:_pinA(_A), _pinB(_B), _pinI(_I), _ppr(ppr), ang_unit(_ang_unit), angv_unit(_angv_unit)
{
   _pinA.mode(PullUp);
   _pinB.mode(PullUp);
   if(_I != NC)
   {
       _pinI.mode(PullUp);
   }

   _eMode = eMode;
   _bIndexTrigger = false;
   _counter = 0;
   _state = 0;
   _nSpeedLastTimer = 0;
   _nSpeedAvrTimeSum = 0;
   _nSpeedAvrTimeCount = -1;
   _fLastSpeed = 0;
   _nSpeedTimeoutMax = 10;
   _nSpeedTimeoutCount = 0;

   if(_eMode & IRQ)
   {
       _pinA.rise(this, &AMT102::ProcessISR);
       _pinA.fall(this, &AMT102::ProcessISR);
       _pinB.rise(this, &AMT102::ProcessISR);
       _pinB.fall(this, &AMT102::ProcessISR);
   }

   if(_eMode & IRQ_NO_JAMMING)
   {
      ProcessISR();
      _state += S2;
   }

   ProcessISR();
   _counter = 0;

   if(_eMode & ANG_VEL)
   {
       _SpeedTimer.reset();
       _SpeedTimer.start();
   }
}

AMT102::~AMT102()
{
    _pinA.rise(NULL);
    _pinA.fall(NULL);
    _pinB.rise(NULL);
    _pinB.fall(NULL);
}

void AMT102::ProcessISR()
{
    int nLoopCounter = 10;
    int pinA, pinB;
    do
    {
        pinA = _pinA;
        pinB = _pinB;

        _state &= MASK;
        if(pinA)
        {
            _state |= A;
        }
        if(pinB)
        {
            _state |= B;
        }
        _state = _modeLUT[_state];

        if(_state & CHG)
        {
            bool bCounterChange = false;

            if(_state & INC)
            {
                _counter++;
                bCounterChange = true;
            }
            if(_state & DEC)
            {
                _counter--;
                bCounterChange = true;
            }

            if(_eMode & IRQ_NO_JAMMING)
            {
                switch (_state & STATE)
                {
                    case S0:
                        _pinB.rise(NULL);
                        _pinB.fall(NULL);
                        _pinA.rise(this, &AMT102::ProcessISR);
                        break;

                    case S1:
                        _pinA.rise(NULL);
                        _pinA.fall(NULL);
                        _pinB.rise(this, &AMT102::ProcessISR);

                    case S2:
                        _pinB.rise(NULL);
                        _pinB.fall(NULL);
                        _pinA.fall(this, &AMT102::ProcessISR);
                        break;

                    case S3:
                        _pinA.rise(NULL);
                        _pinA.fall(NULL);
                        _pinB.fall(this, &AMT102::ProcessISR);
                }
            }

            if(_eMode & ANG_VEL)
            {
                unsigned int act = _SpeedTimer.read_us();
                unsigned int diff = act - _nSpeedLastTimer;
                _nSpeedLastTimer = act;

                if(_nSpeedAvrTimeCount < 0)
                {
                    _nSpeedAvrTimeCount = 0;
                    _nSpeedAvrTimeSum = 0;
                }
                else
                {
                    if(_state & CW)
                        _nSpeedAvrTimeSum += diff;
                    else
                        _nSpeedAvrTimeSum -= diff;

                    _nSpeedAvrTimeCount++;
                }
            }

            if(_bIndexTrigger && bCounterChange && _pinI != NC && _pinI==1)
            {
                _bIndexTrigger =false;
                fPointerIndexTrigger.call(_counter);
            }

            if(bCounterChange)
            {
                fPointerCounterChange.call(_counter);
                if(_state & DIR)
                {
                    fPointerDirectionChange.call(_counter);
                }
            }
        }
    }while(nLoopCounter-- > 0 && (pinA != _pinA || pinB!= _pinB));
}

void AMT102::set_Angle_Unit(Angle_Unit _unit)
{
    ang_unit = _unit;
}

void AMT102::set_AngularVelocity_Unit(AngularVelocity_Unit _unit)
{
    angv_unit = _unit;
}

float AMT102::getAngle()
{
    _angle = getPositionFactor() * (float)_counter / (float)(4 * _ppr);
    return _angle;
}

float AMT102::getAngularVelocity()
{
    int avrTimeSum = 0;
    int avrTimeCount = 0;

    __disable_irq();
    avrTimeSum  = _nSpeedAvrTimeSum;
    avrTimeCount = _nSpeedAvrTimeCount;
    _nSpeedAvrTimeCount = 0;
    _nSpeedAvrTimeSum = 0;
    __enable_irq();

    if(avrTimeCount == 0)
    {
        if(_nSpeedTimeoutCount++ > _nSpeedTimeoutMax)
            _fLastSpeed *= 0.5f;
        _angularVelocity = _fLastSpeed;
    }

    else if(avrTimeCount < 0 || avrTimeSum == 0)
    {
        _angularVelocity = 0;
        _nSpeedTimeoutCount = 0;
    }

    else
    {
         _angularVelocity = 1000000.0f * getSpeedFactor() / ( (float)avrTimeSum / (float)avrTimeCount );
        _nSpeedTimeoutCount = 0;
    }
    _fLastSpeed = _angularVelocity;
    return _angularVelocity;
}