#ifndef LIDARSPI_H
#define LIDARSPI_H

#include "mbed.h"
#include "typeDef.h"
#include "FunctionPointer.h"

/**  LidarSpi class
 *   This class defines methods and structs to interface with the MLX75320 lidar sensing product
 *
 */

class LidarSpi
{
    public:
        
        // Type used to read/write register values. Here, it is 16 bits
        typedef uint16_t REGTYPE;
        typedef unsigned char uchar;
        
        static const uint16_t CRC_SZ   = 2;   // Bytes in SPI CRC field
        static const uint16_t SHORT_SZ = 8;   // Bytes in short SPI packet
        static const uint16_t LONG_SZ  = 300; // Bytes in long SPI packet
        static const uint16_t HDR_SZ   = 2;   // Bytes in SPI header field
    
        static const uint16_t MAXCH     = 16;  // Number of channels
        static const uint16_t MAXSET    = 2;   // Number of acquisition sets
        static const uint16_t MAXPTCNT  = 64;  // Max number of base points
        static const uint16_t MAXECH    = 64 ; // Max echo count
        static const uint16_t MAXACCLOG = 10;  // Max accumulation log2
        static const uint16_t MAXOVRLOG = 3;   // Max oversampling log2
        static const uint16_t MAXLED    = 16;  // Max LED power
        static const uint16_t MAXACC    = 1<<MAXACCLOG;
        static const uint16_t MAXOVR    = 1<<MAXOVRLOG;
        static const uint16_t MAXTRCLEN = 74*8*MAXCH*2;  // Max length of a trace in bytes (samples * MAXOVR * MAXCH * 2bytes)
        static const uint16_t BYTES_PER_ECH = 24; // Nb of bytes per echo
        static const uint16_t START_DELAY =10;
    
        // Used on header ADDR field to request all channels and as idx argument of GetTrace function
        static const uint16_t ADDR_ALLCH= 0x1F;
        
        
        // Structure for the processed data returned by the sensor
        struct Echo
        {
            uint32_t mDistance;
            uint32_t mAmplitude;
            uint32_t mBase;
            uint16_t mMaxIndex;
            uint8_t   mChannelIndex;
            uint8_t   mValid;
            uint32_t mAmplitudeLowScale;
            uint32_t mSaturationWidth;
        };
    
        /// \enum  eASICREG
        /// \brief Lists all registers. Enum value is directly the register address.
        ///        Naming convention: REG_ + <name> + <L/H> + <#>
        ///        <name> = name of register
        ///        <L/H>  = optional low or high register part
        ///        <#>    = optional number, either for acquisition source 0/1 or index
        typedef enum
        {
            REG_MEACL = 0x00,
            REG_MEACH,
            REG_CONTROL,
            REG_ACQCTL0,
            REG_ACQCTL1,
            REG_PTCNT,
            REG_SCANCTL,
            REG_TRIGCTL,
            REG_DELAY,
            REG_TMPSET,
            REG_TMPSHUT,
            REG_TMPIC,
            REG_TMPSRC,
            REG_GAIN0,
            REG_GAIN1,
            REG_GAIN2,
            REG_GAIN3,
            REG_CHANNEL,
            REG_PWMPERIOD0,
            REG_PWMPERIOD1,
            REG_PWMCOUNT,
            REG_PWMWIDTH0,
            REG_PWMWIDTH1   = REG_PWMWIDTH0 + 8,
            REG_FILTER0     = REG_PWMWIDTH1 + 8,
            REG_FILTER1     = REG_FILTER0   + 16,
            REG_THRNEARL0   = REG_FILTER1   + 16,
            REG_THRNEARH0,
            REG_THRMEDL0,
            REG_THRMEDH0,
            REG_THRFARL0,
            REG_THRFARH0,
            REG_THRNEARL1,
            REG_THRNEARH1,
            REG_THRMEDL1,
            REG_THRMEDH1,
            REG_THRFARL1,
            REG_THRFARH1,
            REG_NOISE0,
            REG_NOISE1,
            REG_OFFSETL0,
            REG_OFFSETH0 = REG_OFFSETL0 + 16,
            REG_OFFSETL1 = REG_OFFSETH0 + 16,
            REG_OFFSETH1 = REG_OFFSETL1 + 16,
            REG_SCALEL0  = REG_OFFSETH1 + 16,
            REG_SCALEH0,
            REG_SCALEL1,
            REG_SCALEH1,
            REG_FACTORY,
            REG_ASIL,
            REG_OTP0,
            REG_OTP1,
            REG_OTP2,
            REG_OTP3,
            REG_OTP4,
            REG_OTP5,
            REG_OTP6,
            REG_OTP7,
            REG_SOFTVER,
            REG_ASICVER,
            REG_WATCHDOGL,
            REG_WATCHDOGH,
            // Wai: Changed from REG_MAX
            REG_MAX    =   0xFFFF   // Number of registers
        } eASICREG;
    
        //Internal type used by low level functions: SPI packet type
        enum PackType
        {
            PACK_RREG = 0,      ///< Read register request
            PACK_WREG,          ///< Write register request
            PACK_RFIRM,         ///< Read firmware request
            PACK_WFIRM,         ///< Write firmware request
            PACK_STATUS_S,      ///< Status short
            PACK_STATUS_L,      ///< Status long
            PACK_RDATA_RESP_S,  ///< Read data response short
            PACK_RDATA_RESP_L,  ///< Read data response long
            PACK_WDATA_L,       ///< Write data long
        };
        //Internal type used by low level functions: SPI status status type
        enum StatusType
        {
            STAT_OK = 0,
            STAT_BUSY,
            STAT_CRC,
            STAT_INVALID_REQ,
            STAT_SEQ_NB,
            STAT_TIMEOUT,
        };
        //Internal type used by low level functions: SPI firmware packet type
        enum FirmType
        {
            FW_OTP = 0,
            FW_PROCESSED,
            FW_RAW,
            FW_PATCH,
            FW_PRELOAD,
            FW_TEST,
        };
    
    
    
/*=====================================================================================================================================    
//=====================================================================================================================================   
  _____       _     _ _         __                  _   _                 
 |  __ \     | |   | (_)       / _|                | | (_)                
 | |__) |   _| |__ | |_  ___  | |_ _   _ _ __   ___| |_ _  ___  _ __  ___ 
 |  ___/ | | | '_ \| | |/ __| |  _| | | | '_ \ / __| __| |/ _ \| '_ \/ __|
 | |   | |_| | |_) | | | (__  | | | |_| | | | | (__| |_| | (_) | | | \__ \
 |_|    \__,_|_.__/|_|_|\___| |_|  \__,_|_| |_|\___|\__|_|\___/|_| |_|___/
                                                                      
//=====================================================================================================================================   
//====================================================================================================================================*/       
    
    
        /** Constructor
        * @param: pin names for mosi, miso, clk, chipselect, dataready, reset, trigger and sample*/        
        LidarSpi(PinName mosi, PinName miso, PinName clk, PinName chipSelect, PinName dr, PinName rs, PinName tr, PinName smpl);
        
        /** SpiSetting
        * Change frequency and SPI mode(clock phase and polarity)
        * @param: frequency in Hz, mode ranging from 0 to 3, pointer to Serial interface for debugging
        * @return: 0 on success        */
        int SpiSetting(long freq, int mode, Serial* pc);
        
        /** ReadReg
        * Read register at address
        * @param: register address, pointer to variable to register valueresult
        * @return: 0 on success        */
        int ReadReg   ( uint32_t reg, uint32_t *val);
        
        /** WriteReg
        * Write register to address
        * @param: register address, value to write
        * @return: 0 on success        */
        int WriteReg  ( uint32_t reg, uint32_t val);
        
                
        /** GetEchoes
        * Perform measurement and read processed data
        * @param: pointer to Echo structure array to store measurement, structure array size (must be >=64)
        * @return: 0 on success        */
        int GetEchoes ( Echo *ech, uint16_t maxN);
        
        /** GetEchoes
        * Perform measurement and read processed data
        * @param: pointer to Echo structure array to store measurement, structure array size (must be >=64), pointer to serial interface for debugging
        * @return: 0 on success        */
        int GetEchoes ( Echo *ech, uint16_t maxN, Serial* pc);
        
        /** GetTrace
        * Perform measurement and read raw trace buffer
        * @param: pointer to int array to store trace buffer, structure array size (must be >=9472*2), pointer to serial interface for debugging
        * @return: 0 on success        */
        int GetTrace  ( uint16_t *buf, uint16_t maxN, Serial* pc);
        
        /** LoadPatch
        * Load patch stored on mbed Flash memory in MLX75320 and enable patch
        * @param: path of patch.hex location, , pointer to serial interface for debugging
        * @return: 0 on success       */
        int LoadPatch (const char *patch, Serial *pc);
        
        /** SetTrace
        * Set a variety of registers according to the preset values for a raw data measurement
        * @return: 0 on success       */
        int setTrace(void);
        
        /** SetTrace
        * Set a variety of registers according to the preset values for a processed data measurement
        * @return: 0 on success       */
        int setEcho(void);
        
        /** SetLed
        * Set a variety of registers according to the preset values to either fire or don't fire the LEDs during a measurement
        * @param: True to enable LEDs, False to disable       */
        int setLed(bool state);
        
        
/*=====================================================================================================================================    
//=====================================================================================================================================   
  _____       _                        _                    _      _                       _                __                  _   _                 
 |_   _|     | |                      | |                  | |    | |                     (_)              / _|                | | (_)                
   | |  _ __ | |_ ___ _ __ _ __   __ _| |   ___  _ __    __| | ___| |__  _   _  __ _  __ _ _ _ __   __ _  | |_ _   _ _ __   ___| |_ _  ___  _ __  ___ 
   | | | '_ \| __/ _ \ '__| '_ \ / _` | |  / _ \| '__|  / _` |/ _ \ '_ \| | | |/ _` |/ _` | | '_ \ / _` | |  _| | | | '_ \ / __| __| |/ _ \| '_ \/ __|
  _| |_| | | | ||  __/ |  | | | | (_| | | | (_) | |    | (_| |  __/ |_) | |_| | (_| | (_| | | | | | (_| | | | | |_| | | | | (__| |_| | (_) | | | \__ \
 |_____|_| |_|\__\___|_|  |_| |_|\__,_|_|  \___/|_|     \__,_|\___|_.__/ \__,_|\__, |\__, |_|_| |_|\__, | |_|  \__,_|_| |_|\___|\__|_|\___/|_| |_|___/
                                                                                __/ | __/ |         __/ |                                             
                                                                               |___/ |___/         |___/                                             
//=====================================================================================================================================   
//====================================================================================================================================*/    
        
        
        
        /** ReadReg (overloaded)
        * Read register at address
        * @param: register address, pointer to variable to register valueresult, pointer to serial interface for debugging
        * @return: 0 on success        */
        int ReadReg(uint32_t reg, uint32_t *val, Serial* pc);
        
        /** WriteReg
        * Write register to address
        * @param: register address, value to write, pointer to serial interface for debugging
        * @return: 0 on success        */
        int WriteReg  ( uint32_t reg, uint32_t val, Serial* pc);
        
        /** GetTraceOne
        * Perform measurement and read raw trace buffer, this method only reads back a single SPI packet
        * @param: pointer to int array to store trace buffer, structure array size (must be >=9472*2), pointer to serial interface for debugging
        * @return: 0 on success        */
        int GetTraceOne  ( uint16_t *buf, uint16_t maxN, uint16_t nSam, uint16_t idx,int index , Serial* pc);
        
        /** Trigger
        * Change pin to high or low to use as trigger point for oscilloscope
        * @param: 1 or 0 to set voltage level   */
        void Trigger(int level);
        
        
        int PrintAllReg (uint16_t * regs, uint32_t * val, uint16_t size);
        
        int TxPacket(uint8_t* rData, uint16_t *rSz, uint8_t *tData, uint16_t tSz);   
        int TxPacketWord(uint8_t* rData, uint16_t *rSz, uint8_t *tData, uint16_t tSz);   
        int TxPacket(uint8_t* rData, uint16_t *rSz, uint8_t *tData, uint16_t tSz, Serial* pc);   
        int TxPacketSlow(uint8_t* rData, uint16_t *rSz, uint8_t *tData, uint16_t tSz, uint16_t usDelay);
        int BasicRead();
        int BasicTransfer(uint8_t* rData, uint16_t rSz, uint8_t *wData, uint16_t wSz, const event_callback_t callback);
        int WriteRegSpeed  ( uint32_t reg, uint32_t val, uint16_t usDelay);
        int WriteRegSpeed  ( uint32_t reg, uint32_t val, uint16_t usDelay, Serial* pc);
        
        //int SetConfig ( int configNum);
        //int SetAcqCfg ( uint16_t set, uint16_t led, uint16_t accLog, uint16_t ovrLog);
        //int Acquire   ( uchar ab, uchar evGain, uchar evCh, uchar odGain, uchar odCh);
        //int GetFrame  ( uint16_t *buf, uint16_t maxN, uint16_t nSam, uint16_t nFrm);
        
    
        
        
    private:
        SPI device;
        DigitalOut chipS;
        DigitalIn dataReady;
        DigitalOut resetPin;
        DigitalOut trigger;
        DigitalIn sampling;
        
        int parse_hex_line(char *theline, uint8_t bytes[], uint16_t *addr, uint16_t *num, uint16_t *code);
        int LoadPatchFragment(const char *patch, uint16_t *addrStart, uint16_t *startLine, uint16_t *nBytes, uint8_t *memory, Serial* pc);

          
};


#endif