// ****************************************************************************
/*!
Module:     Melexis Base SPI

Platform:   Win32 - Win64 - embedded

\file       MLX_BaseSPI.h

\brief      Base control module the ASIC SPI, Melexis protocol.

\author     Jean-F. Bernier

\since      2015-04-23
*/
// ***************************************************************************

#ifndef _MLX_BASESPI_H_
#define _MLX_BASESPI_H_

#ifdef __cplusplus
extern "C" {
#endif

//*****************************************************************************
//*************** Header includes *********************************************
//*****************************************************************************
#include "typeDef.h"

//*****************************************************************************
//*************** Constants and Macros ****************************************
//*****************************************************************************
#define MLX_CRC_SZ      2    /// Bytes in SPI CRC field
#define MLX_SHORT_SZ    8    /// Bytes in short SPI packet
#define MLX_LONG1_SZ     300  /// Bytes in long SPI packet; BSI spec raw_data packet size
#define MLX_HDR_SZ      2    /// Bytes in SPI header field
#define MLX_LONG2_SZ     260  /// Bytes in ECHO packet

// Bytes in payload field
#define MLX_SHORT_DATA_SZ   (MLX_SHORT_SZ - MLX_HDR_SZ - MLX_CRC_SZ)
#define MLX_LONG1_DATA_SZ    (MLX_LONG1_SZ - MLX_HDR_SZ - MLX_CRC_SZ)
#define MLX_LONG2_DATA_SZ    (MLX_LONG2_SZ - MLX_HDR_SZ - MLX_CRC_SZ)

#if defined(WIN32) || defined(_WIN64)
    #ifdef SPI_EXPORTS
        #define SPI_DLL __declspec(dllexport)
    #else
        #define SPI_DLL __declspec(dllimport)
    #endif
#else
    #define SPI_DLL
#endif



// Basic computation of the number of elements of a STATIC array
#define ARRAY_SZ(a)     (sizeof(a)/sizeof(a[0]))

#ifndef BIT
// Constructs a mask of a single bit value. "OR" to make a mask: mask = BIT(3) | BIT(6) | BIT(12)
#define BIT(x)                  ( 1<<(x) )
#endif

// Set to 1, clear to 0, obtain value or flip a single bit in variable
#ifndef BIT_SET
#define BIT_SET(var, bit)     ( (var) |=  BIT(bit) )
#endif

#ifndef BIT_CLR
#define BIT_CLR(var, bit)     ( (var) &= ~BIT(bit) )
#endif

#ifndef BIT_GET
#define BIT_GET(var, bit)     ( (var) &   BIT(bit) )
#endif

#ifndef BIT_FLIP
#define BIT_FLIP(var, bit)    ( (var) ^=  BIT(bit) )
#endif

// Set to 1, clear to 0, obtain value or flip a variable at specified bit MASK positions
#ifndef MASK_SET
#define MASK_SET(var, mask)   ( (var) |=  (mask) )
#endif

#ifndef MASK_CLR
#define MASK_CLR(var, mask)   ( (var) &= ~(mask) )
#endif

#ifndef MASK_GET
#define MASK_GET(var, mask)   ( (var) &   (mask) )
#endif

#ifndef MASK_FLIP
#define MASK_FLIP(var, mask)  ( (var) ^=  (mask) )
#endif
//*****************************************************************************
//*************** Data Type Definitions ***************************************
//*****************************************************************************

/// \enum   FirmType
/// \brief  Lists all firmware access types.
typedef enum
{
    FW_OTP = 0,
    FW_PROCESSED,
    FW_RAW,
    FW_PATCH,
    FW_PRELOAD,
    FW_TEST,
} FirmType;


/// \union  PACK_SHORT
/// \brief  Defines a short packet.
typedef union
{
    // Main data buffer
    uint8 buf[MLX_SHORT_SZ];

    // Structured data: header, payload and CRC
    struct
    {
        uint16 hdr;
        uint8  data[MLX_SHORT_DATA_SZ];
        uint16 crc;
    };
} PACK_SHORT;


///
/// MLXSPI uses 2 sized "long packets": 
/// Trace data is trasmitted in 148 word long packets (LONG1)
/// Echo data and patch data are transmitted in 128 word packets (LONG2)
//
/// \union  PACK_LONG1
/// \brief  Defines a long packet.
typedef union
{
    // Main data buffer
    uint8 buf[MLX_LONG1_SZ];

    // Structured data: header, payload and CRC
    struct
    {
        uint16 hdr;
        uint8  data[MLX_LONG1_DATA_SZ];
        uint16 crc;
    };
} PACK_LONG1;

/// \union  PACK_LONG2
/// \brief  Defines a long packet.
typedef union
{
	// Main data buffer
	uint8 buf[MLX_LONG2_SZ];

	// Structured data: header, payload and CRC
	struct
	{
		uint16 hdr;
		uint8  data[MLX_LONG2_DATA_SZ];
		uint16 crc;
	};
} PACK_LONG2;


/// \enum   PackType
/// \brief  Lists all packet types.
typedef enum
{
    PACK_RREG = 0,      ///< Read register request
    PACK_WREG,          ///< Write register request
    PACK_RFIRM,         ///< Read firmware request
    PACK_WFIRM,         ///< Write firmware request
    PACK_STATUS_S,      ///< Status short
    PACK_STATUS_L,      ///< Status long
    PACK_RDATA_RESP_S,  ///< Read data response short
    PACK_RDATA_RESP_L,  ///< Read data response long
    PACK_WDATA_L,       ///< Write data long
} PackType;


/// \enum   StatusType
/// \brief  Lists all error statuses.
typedef enum
{
    STAT_OK = 0,
    STAT_BUSY,
    STAT_CRC,
    STAT_INVALID_REQ,
    STAT_SEQ_NB,
    STAT_TIMEOUT,
} StatusType;

/// \struct MLX_PackNfo
/// \brief  Structure holding various data for packet encoding / decoding.
typedef struct
{
    PackType    pktype; ///< Packet type
    uint8       size;   ///< Size field
    uint8       seq;    ///< Sequence number
    FirmType    fwtype; ///< Firmware access type
    uint16      addr;   ///< Address field (register or firmware block)
    uint32      data;   ///< Useful data (register or firmware value)
    const uint8 *dataP; ///< Data pointer for long packets (firmware write)
    uint8       iserr;  ///< Is error state for status packet
    uint16      error;  ///< Error qualifier for status packet
} MLX_PackNfo;

//*****************************************************************************
//*************** Public Function Declarations ********************************
//*****************************************************************************

// *********************
// *** Master functions
// *********************
int  MLX_DecodeResS ( const PACK_SHORT *pack, uint32 *val);
int  MLX_DecodeResL1 ( const PACK_LONG1  *pack);
int  MLX_DecodeResL2 ( const PACK_LONG2  *pack);


int  MLX_ReqReadReg ( PACK_SHORT *pack, uint16 reg);
int  MLX_ReqWriteReg( PACK_SHORT *pack, uint16 reg, uint16 val);
int  MLX_ReqReadTrc ( PACK_SHORT *pack);
int  MLX_ReqReadEch ( PACK_SHORT *pack);
int  MLX_ReqWriteFW ( PACK_SHORT *pack, uint size, uint16 data);

int  MLX_WriteDataL2 ( PACK_LONG2 *pack, uint size, uint seq, const uint8 *data);

// *********************
// *** Slave functions
// *********************
int  MLX_DecodeReqS ( const PACK_SHORT *pack, MLX_PackNfo *nfo);
int  MLX_ResReadReg ( PACK_SHORT *pack, uint16 val);
int  MLX_ResReadDataL1( PACK_LONG1 *pack, uint size, uint seq, const uint8 *data);
int  MLX_ResReadDataL2( PACK_LONG2 *pack, uint size, uint seq, const uint8 *data);

// *********************
// *** Generic functions
// *********************
int MLX_EncodeStatusS( PACK_SHORT *pack, uint8 iserr, uint16 err);
int MLX_DecodeStatusS( const PACK_SHORT *pack, uint8 *iserr, uint16 *err);
int MLX_EncodeStatusL1( PACK_LONG1 *pack, uint8 iserr, uint16 err);
int MLX_DecodeStatusL1( const PACK_LONG1 *pack, uint8 *iserr, uint16 *err);
int MLX_EncodeStatusL2( PACK_LONG2 *pack, uint8 iserr, uint16 err);
int MLX_DecodeStatusL2( const PACK_LONG2 *pack, uint8 *iserr, uint16 *err);

SPI_DLL int    SetBitVector( uint8 *buf, uint32 off, uchar wid, uint32 val, uchar inv);
SPI_DLL int    GetBitVector( const uint8 *buf, uint32 off, uchar wid, uint32 *val, uchar inv);
SPI_DLL uint32 ReverseBits ( uint32 bits, char nb);

#ifdef __cplusplus
}
#endif

#endif // _MLX_BASESPI_H_

