#include "sample_hardware.hpp"

#define RED_DONE 1
#define YELLOW_DONE 2

//Digital outputs
DigitalOut onBoardLED(LED1);
DigitalOut redLED(PE_15);
DigitalOut yellowLED(PB_10);
DigitalOut greenLED(PB_11);

//Inputs
DigitalIn  onBoardSwitch(USER_BUTTON);
DigitalIn  SW1(PE_12);
DigitalIn  SW2(PE_14);
AnalogIn 	adcIn(PA_0);

//Environmental Sensor driver
#ifdef BME
BME280 sensor(D14, D15);
#else
BMP280 sensor(I2C_SDA, I2C_SCL);
#endif

//LCD Driver (provided via mbed repository)
//RS D9
//E  D8
//D7,6,4,2 are the 4 bit for d4-7
TextLCD lcd(D9, D8, D7, D6, D4, D2); // rs, e, d4-d7

//SD Card
SDBlockDevice sd(PB_5, D12, D13, D10); // mosi, miso, sclk, cs

sensorData buffer[BUFFERSIZE];

FATFileSystem* fs;

//POWER ON SELF TEST
void post() 
{
	
		seconds = time(NULL);
		timeData = localtime(&seconds);
		set_time(mktime(timeData));	
	
    //POWER ON TEST (POT)
    pc->printf("**********STARTING POWER ON SELF TEST (POST)**********\n\r");
    
    //Test LEDs
    pc->printf("ALL LEDs should be blinking\n\r");
    for (unsigned int n=0; n<10; n++) {
        redLED    = 1;
        yellowLED = 1;
        greenLED  = 1;
        wait(0.05);
        redLED    = 0;
        yellowLED = 0;
        greenLED  = 0;     
        wait(0.05);         
    } 
    
    //Output the switch states (hold them down to test)
    pc->printf("SW1: %d\tSW2: %d\n\r", SW1.read(), SW2.read());    
    pc->printf("USER: %d\n\r", onBoardSwitch.read()); 
    
    //Output the ADC
    pc->printf("ADC: %f\n\r", adcIn.read());
    
    //Read Sensors (I2C)
    float temp = sensor.getTemperature();
    float pressure = sensor.getPressure();
    #ifdef BME
    float humidity = sensor.getHumidity();
    #endif
   
    //Display in PuTTY
    pc->printf("Temperature: %5.1f\n\r", temp);
    pc->printf("Pressure: %5.1f\n\r", pressure);
    #ifdef BME
    pc->printf("Pressure: %5.1f\n\r", humidity);
    #endif
    
    //Display on LCD
    redLED = 1;
    lcd.cls();
    lcd.printf("DISPLAY TEST...");
    wait(0.5);
    redLED = 0;
    pc->puts("**********POST END**********\r\n\n");
    
    pc->puts("READY FOR COMMANDS\n\r");
    pc->puts("READ ALL/<N>    	: Read samples\r\n");
    pc->puts("DELETE ALL/<N>  	: Delete samples\r\n");
    pc->puts("SETDATE DDMMYYYY	: Set the date\r\n");
    pc->puts("SETTIME HHMMSS  	: Set the time\r\n");
    pc->puts("SETT <T>        	: Set the sampling rate (15s)\r\n");
    pc->puts("STATE <X>       	: Toggle sampling on/off (off)\r\n");
    pc->puts("LOGGING <X>     	: Toggle logging on/off (off)\r\n\n");
 
		pc->puts("PLEASE UPDATE DATE AND TIME IF REQUIRED\r\n");
		pc->printf("CURRENT DATE AND TIME : %s\r\n", ctime(&seconds));
}

void errorCode(ELEC350_ERROR_CODE err)
{
    switch (err) {
      case OK:
        greenLED = 1;
        wait(1.0);
        greenLED = 0;
        return;                
      case FATAL:
        while(1) {
            redLED = 1;
            wait(0.1);
            redLED = 0;
            wait(0.1);                
        }
    }
}

void sampleProducer()
{
    while(true)
    {
        //High priority thread 
        Thread::signal_wait(TAKE_SAMPLE);
    
        Nspaces = spaceAvailable.wait(0); //Non-blocking
        bufferLock.lock();
        producer_tout.attach(producer_toutISR, TOUT_TIME_DEF);
               
        //Update buffer           
        if ((newestIndex == oldestIndex) && (Nspaces==0))
        {
            oldestIndex = (oldestIndex+1) % BUFFERSIZE;     
        }
            
        newestIndex = (newestIndex+1) % BUFFERSIZE;  //CIRCULAR 
         
        double temp_r = sensor.getTemperature();
        double press_r = sensor.getPressure();
        float light_r = adcIn.read();
        
        buffer[newestIndex].updatetemp(temp_r);
        buffer[newestIndex].updatepress(press_r);
        buffer[newestIndex].updatelight(light_r);
        buffer[newestIndex].updateTime();
               
        if (Nspaces != 0)
        {   
            Nspaces--;            
        } 
          
        samplesInBuffer.release();

        if(logging)
        {
            printQueue.call(printf, "Sample placed in buffer at position %d\r\nNumber of spaces available in buffer:%d\r\n\n", newestIndex, Nspaces); 
        }
        
        bufferLock.unlock();
        producer_tout.detach();
    }
}

void sampleConsumer()
{
    while(true)
    {
        //write to the SD card from oldestindex up to newestIndex.
        
        Nsamples = samplesInBuffer.wait(); //Block if no samples to take - acquires
        
        bufferLock.lock();
        consumer_tout.attach(consumer_toutISR,TOUT_TIME_DEF);
        if (sd_init)
        {      
                oldestIndex = (oldestIndex+1) % BUFFERSIZE;                        
                SDqueue.call(SDaddSample,buffer[oldestIndex].getTime(), buffer[oldestIndex].gettemp(), buffer[oldestIndex].getpress(), buffer[oldestIndex].getlight(), oldestIndex);                
        }      
        else
        {
            samplesInBuffer.release();   
        } 
        bufferLock.unlock();
        consumer_tout.detach();     
    }       
    
}

void sampleISR()
{
    producer_thread.signal_set(TAKE_SAMPLE);   
}

void producer_toutISR(void)
{
    threadstates |= PRODUCER;
}

void consumer_toutISR(void)
{
    threadstates |= CONSUMER;   
}

/*SensorData Class Definitions*/

sensorData::sensorData(void)
{
		this->temperature = 0;
		this->pressure = 0;
		this->lightlevel = 0;
		this->time_str = "";
}
void sensorData::updateTime()
{
		string t;
		time_t seconds; 
		timeLock.lock();
		seconds = time(NULL);            
		t = ctime(&seconds);
		timeLock.unlock();
		t = t.substr(0,t.length()-1);
		this->time_str = t; 
		
}
void sensorData::updatetemp(double t) {this->temperature = t;}
void sensorData::updatepress(double p) {this->pressure = p;}
void sensorData::updatelight(float l) {this->lightlevel = l;}
double sensorData::gettemp(void) {return this->temperature;}
double sensorData::getpress(void) {return this->pressure;}
float sensorData::getlight(void) {return this->lightlevel;}  
string sensorData::getTime(void) {return this->time_str;}

