/* mbed Microcontroller Library
 * Copyright (c) 2006-2012 ARM Limited
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#include "CAN.h"

#if DEVICE_CAN

#include "cmsis.h"

namespace mbed {

CAN::CAN(PinName rd, PinName td) {
    can_init(&_can, rd, td);
}

CAN::~CAN() {
    can_free(&_can);
}
  
int CAN::frequency(int f) {
    return can_frequency(&_can, f);
}

int CAN::write(CANMessage msg) {
    return can_write(&_can, msg, 0);
}
    
int CAN::read(CANMessage &msg) {
    return can_read(&_can, &msg);
}

void CAN::reset() {
    can_reset(&_can);
}

unsigned char CAN::rderror() {
    return can_rderror(&_can);
}

unsigned char CAN::tderror() {
    return can_tderror(&_can);
}

void CAN::monitor(bool silent) {
    can_monitor(&_can, (silent) ? 1 : 0);
}

static FunctionPointer* can_obj[2] = { NULL };

// Have to check that the CAN block is active before reading the Interrupt
// Control Register, or the mbed hangs
void can_irq(void) {
    uint32_t icr;
    
    if(LPC_SC->PCONP & (1 << 13)) {
        icr = LPC_CAN1->ICR;
        
        if(icr && (can_obj[0] != NULL)) {
            can_obj[0]->call();
        }
    }
    
    if(LPC_SC->PCONP & (1 << 14)) {
        icr = LPC_CAN2->ICR;
        if(icr && (can_obj[1] != NULL)) {
            can_obj[1]->call();
        }
    }

}

void CAN::setup_interrupt(void) {
    switch ((int)_can.dev) {
        case CAN_1: can_obj[0] = &_rxirq; break;
        case CAN_2: can_obj[1] = &_rxirq; break;
    }
    _can.dev->MOD |= 1;
    _can.dev->IER |= 1;
    _can.dev->MOD &= ~1;
    NVIC_SetVector(CAN_IRQn, (uint32_t) &can_irq);
    NVIC_EnableIRQ(CAN_IRQn);
}

void CAN::remove_interrupt(void) {
    switch ((int)_can.dev) {
        case CAN_1: can_obj[0] = NULL; break;
        case CAN_2: can_obj[1] = NULL; break;
    }
    
    _can.dev->IER &= ~(1);
    if ((can_obj[0] == NULL) && (can_obj[1] == NULL)) {
        NVIC_DisableIRQ(CAN_IRQn);
    }
}

void CAN::attach(void (*fptr)(void)) {
    if (fptr != NULL) {
        _rxirq.attach(fptr);
        setup_interrupt();
    } else {
        remove_interrupt();
    }
}

} // namespace mbed

#endif
