#include "mbed.h"
#include "mbed_events.h"
#include "USBMIDI.h"
#include "midiswitch.h"


/* ******************** Configuration ***************************** */

#define NUM_SWITCHES 2

/* array of SwitchConfig structs */
SwitchConfig switches[NUM_SWITCHES] = {
    // pin, type, channel, data1, on_value, off_value
    {PB_4, 0xB, 0, 1, 127, 0},
    {PB_5, 0xC, 0, 0, 1, -1},
};


/* ******************** End Configuration ************************* */

#ifndef NDEBUG
Serial *serial;
#endif
USBMIDI * midi;
/* array of pointers to SwitchHandler instances */
SwitchHandler * handlers[NUM_SWITCHES];


void write_midi_msg(MIDIMessage msg) {
#ifndef NDEBUG
    serial->printf("Sending MIDI message controller=%d channel=%d value=%d\r\n",
                   msg.controller(), msg.channel(), msg.value());
#endif
    midi->write(msg);
}

void handle_sysex(MIDIMessage msg) {
    uint8_t sw;
#ifndef NDEBUG
    serial->printf("MIDI recv: header=0x%02X status=0x%02X data1=0x%02X data2=%02d length=%d\r\n",
                   msg.data[0], msg.data[1], msg.data[2], msg.data[3], msg.length);
#endif

    if (msg.data[1] == 0xF0 && msg.data[2] == 0x7D && msg.data[11] == 0xF7) {
        sw = msg.data[3];
        if (sw < NUM_SWITCHES) {
            switches[sw].type = msg.data[4] & 0xF;
            switches[sw].channel = msg.data[5] & 0xF;
            switches[sw].data1 = msg.data[6] & 0x7F;
            if (msg.data[7] > 0)
                switches[sw].on_value = msg.data[8] & 0x7F;
            else
                switches[sw].on_value = -1;
            if (msg.data[9] > 0)
                switches[sw].off_value = msg.data[10] & 0x7F;
            else
                switches[sw].off_value = -1;
            handlers[sw]->setConfig(switches[sw]);
#ifndef NDEBUG
            serial->printf("Changed configuration for switch %d.\r\n", sw + 1);
            serial->printf("type: 0x%02X\r\n", switches[sw].type);
            serial->printf("channel: %02d\r\n", switches[sw].channel);
            serial->printf("data1: %02d\r\n", switches[sw].data1);
            serial->printf("on_value: %02d\r\n", switches[sw].on_value);
            serial->printf("off_value: %02d\r\n", switches[sw].off_value);
#endif
        }
    }
}

int main() {
#ifndef NDEBUG
    serial = new Serial(PA_9, PA_10);
    serial->printf("Creating event queue...\r\n");
#endif
    EventQueue queue;

#ifndef NDEBUG
    serial->printf("Creating USBMIDI device...\r\n");
#endif
    midi = new USBMIDI(0x1f00, 0x2012, 0x0001);

#ifndef NDEBUG
    serial->printf("Initializing LED...\r\n");
#endif
    DigitalOut led1(LED1);
    led1 = 1;

#ifndef NDEBUG
    serial->printf("Initializing event queue thread...\r\n");
#endif
    // create a thread that'll run the event queue's dispatch function
    Thread usbThread;
#ifndef NDEBUG
    serial->printf("Starting event queue thread...\r\n");
#endif
    usbThread.start(callback(&queue, &EventQueue::dispatch_forever));

    for (int sw=0; sw < NUM_SWITCHES; sw++) {
#ifndef NDEBUG
        serial->printf("Initializing switch handler %d...\r\n", sw + 1);
#endif
        handlers[sw] = new SwitchHandler(&queue, &write_midi_msg, switches[sw]);
    }

    midi->attach(&handle_sysex);

#ifndef NDEBUG
    serial->printf("Entering main loop...\r\n");
#endif
    while (true) {
        wait(0.5f);
        led1 = !led1;
    }
}
