/**
 * Simple MAG3110 test/demo program, based on various bits of code found.
 * Updated for Mbed FRDM-KL25Z and Avnet Wi-Go module.
 *
 * Operation:
 * 1. On startup Red LED flashes indicating calibration mode entered
 * 2. Slide finger along capacitive sensor and release
 * 3. Green LED flashes indicating calibration mode.
 * 4. Rotate board once in horizontal plane
 * 5. Tap and release capacitive sensor. Board now calibrated with min/max values
 * 6. LEDs now off. Rotate board. When Blue LED lights the bottom of the board is
 *    pointing to approximately North (+/- 22.5')
 *
 * If USB cable connected then you would see debug messages and other headings displayed, e.g. N, NE, E, SE, S, SW, W and NW.
 *
 * Also displays register values when starting.
 *
 * Ideally the calibration settings would be stored in flash/eeprom and retrieves each time.
 *
 * By Andrew D. Lindsay, @AndrewDLindsay
 *
 *
 *
 */

#include "mbed.h"
#include "TSISensor.h"
#include "math.h"
#include "MAG3110.h"


#define ON  0
#define OFF 1

// Some LEDs for showing status
DigitalOut redLed(LED_RED);
DigitalOut greenLed(LED_GREEN);
DigitalOut blueLed(LED_BLUE);

// Slide sensor acts as a button
TSISensor tsi;

//MAG3100 mag(PTE0, PTE1, &pc); //DEBUG verion
MAG3110 mag(PTE0, PTE1);



void calXY() //magnetometer calibration: finding max and min of X, Y axis
{
    int tempXmax, tempXmin, tempYmax, tempYmin, newX, newY;

    redLed = ON;

    printf("Waiting for initial press\n");
    // Wait for slider to be pressed
    while( tsi.readDistance() == 0 ) {
        redLed = ON;
        wait(0.2);
        redLed = OFF;
        wait(0.2);
    }

    printf("Waiting for release\n");

    // Wait for release
    while( tsi.readDistance() != 0 ) {
        redLed = OFF;
        wait(0.2);
        redLed = ON;
        wait(0.2);
    }
    redLed = OFF;
    wait(0.5);

    printf("Rotate\n");

    tempXmax = tempXmin = mag.readVal(MAG_OUT_X_MSB);
    tempYmax = tempYmin = mag.readVal(MAG_OUT_Y_MSB);

    while(tsi.readDistance() == 0) {
        greenLed = ON;
        wait(0.1);
        greenLed = OFF;
        wait(0.1);
        newX = mag.readVal(MAG_OUT_X_MSB);
        newY = mag.readVal(MAG_OUT_Y_MSB);
        if (newX > tempXmax) tempXmax = newX;
        if (newX < tempXmin) tempXmin = newX;
        if (newY > tempYmax) tempYmax = newY;
        if (newY < tempYmin) tempYmin = newY;
    }
   
    mag.setCalibration( tempXmin, tempXmax, tempYmin, tempYmax );

    // Wait for release
    while( tsi.readDistance() != 0 ) {
        greenLed = OFF;
        wait(0.2);
        greenLed = ON;
        wait(0.2);
    }
    greenLed = OFF;
    wait(1.0);
}

int main()
{
    printf("MAG3110 Test\n");

    redLed = OFF;
    greenLed = OFF;
    blueLed = OFF;

    // Get some values
    printf("DR_STATUS %X\n", mag.readReg( MAG_DR_STATUS ));
    printf("WHO_AM_I %X\n", mag.readReg( MAG_WHO_AM_I ));
    printf("SYSMOD %X\n", mag.readReg( MAG_SYSMOD ));
    printf("DIE_TEMP %d\n", mag.readReg( MAG_DIE_TEMP ));

    printf("OFF_X %d\n", mag.readVal( MAG_OFF_X_MSB ));
    printf("OFF_Y %d\n", mag.readVal( MAG_OFF_Y_MSB ));
    printf("OFF_Z %d\n", mag.readVal( MAG_OFF_Z_MSB ));

    printf("CTRL_REG1 %X\n", mag.readReg( MAG_CTRL_REG1 ));
    printf("CTRL_REG2 %X\n", mag.readReg( MAG_CTRL_REG2 ));

    printf("calibrate\n");
    calXY();
    printf("....Finished\n");

    redLed = OFF;
    greenLed = OFF;
    blueLed = OFF;

    while (1) {
        wait(0.5);
        int xVal = mag.readVal(MAG_OUT_X_MSB);
        int yVal = mag.readVal(MAG_OUT_Y_MSB);
        float heading = mag.getHeading();

        // Do something with heading - display direction and turn on blue LED if heading approx north
        if (abs(heading) <= 22.5) {
            printf("N\n");
            blueLed = ON;
        } else blueLed = OFF;
        if (abs(heading) >= 157.5) printf("S\n");
        if (heading >= 67.5 && heading <= 112.5) printf("E \n");
        if (heading <= -67.5 && heading >= -112.5) printf("W \n");
        if (heading > 22.5 && heading < 67.5) printf("NE\n");
        if (heading < -22.5 && heading > -67.5) printf("NW\n");
        if (heading > 112.5 && heading < 157.5) printf("SE\n");
        if (heading < -112.5 && heading > -157.5) printf("SW\n");

        if (heading < 0) heading += 360.0;
        printf("X = %d, Y = %d, Heading %f\n", xVal, yVal, heading);

    }
}
