/*
* N3310LCD. A program to interface mbed with the nuelectronics
* Nokia 3310 LCD shield from www.nuelectronics.com. Ported from
* the nuelectronics Arduino code.
*
* Copyright (C) <2009> Petras Saduikis <petras@petras.co.uk>
*
* Converted to a mbed library by Andrew D. Lindsay
*
* This file is part of N3310LCD.
*
* N3310LCD is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* N3310LCD is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with N3310LCD.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "N3310SPIConfig.h"
#include "N3310LCD.h"
#include "Joystick.h"
#include "mbed_bmp.h"
#include "pacman.h"
#include "bitmap.h"

// demo for nuelectronics Nokia 3310 LCD shield (www.nuelectronics.com)
//

// menu starting points
#define MENU_X    10        // 0-83
#define MENU_Y    0        // 0-5

#define DEMO_ITEMS 5

void temperature(N3310LCD* lcd, Joystick* jstick);
void charmap(N3310LCD* lcd, Joystick* jstick);
void bitmap(N3310LCD* lcd, Joystick* jstick);
void about(N3310LCD* lcd, Joystick* jstick);
void graphics(N3310LCD* lcd, Joystick* jstick);

// menu definition
char menu_items[DEMO_ITEMS][12] = {
    "TEMPERATURE",
    "CHAR MAP",
    "BITMAP",
    "GRAPHICS",
    "ABOUT"
};

void (*menu_funcs[DEMO_ITEMS])(N3310LCD*,Joystick*) = {
    temperature,
    charmap,
    bitmap,
    graphics,
    about
};

void temperature(N3310LCD* lcd, Joystick* jstick)
{
    lcd->writeStringBig(5, 1, "+21.12", NORMAL);
    lcd->writeString(73, 2, "C", NORMAL);
}

void charmap(N3310LCD* lcd, Joystick* jstick)
{
    for(int i = 0; i < 5; i++) {
        for(int j = 0; j < 14; j++) {
            lcd->locate(j*6, i);
            lcd->writeChar(i*14 + j + 32, NORMAL);
        }
    }
}

void bitmap(N3310LCD* lcd, Joystick* jstick)
{
    for( int i=0; i<10; i++ ) {
        lcd->drawBitmap(20, 1, mbed_bmp, 48, 24);
        wait(1);
        lcd->clearBitmap(20, 1, 48, 24);
        lcd->drawBitmap(0, 0, fy1, 84, 48);
        wait(1);
        lcd->clearBitmap(0, 0, 84, 48);
    }
    lcd->drawBitmap(20, 1, mbed_bmp, 48, 24);
}

void about(N3310LCD* lcd, Joystick* jstick)
{
    lcd->writeString(0, 1, "Nokia 3310 LCD", NORMAL);
    lcd->writeString(15, 2, "driven by", NORMAL);
    lcd->writeString(10, 3, "KL25Z mbed", NORMAL);
}



void initMenu(N3310LCD* lcd)
{
    lcd->writeString(MENU_X, MENU_Y, menu_items[0], HIGHLIGHT );

    for (int i = 1; i < DEMO_ITEMS; i++) {
        lcd->writeString(MENU_X, MENU_Y + i, menu_items[i], NORMAL);
    }
}

void waitforOKKey(N3310LCD* lcd, Joystick* jstick)
{
    lcd->writeString(38, 5, "OK", HIGHLIGHT );

    int key = 0xFF;
    while (key != CENTER_KEY) {
        for (int i = 0; i < NUM_KEYS; i++) {
            if (jstick->getKeyState(i) !=0) {
                jstick->resetKeyState(i);  // reset
                if (CENTER_KEY == i) key = CENTER_KEY;
            }
        }
    }
}

// Check if joystick is moved or pressed
uint8_t checkKeypressed( Joystick* jstick)
{
    uint8_t key = 0xFF;

    for(int i=0; i<NUM_KEYS; i++) {
        if (jstick->getKeyState(i) !=0) {
            jstick->resetKeyState(i);  // reset
            if (CENTER_KEY == i) key = CENTER_KEY;
        }
    }
    return key;
}

/*
void autoDemo(N3310LCD* lcd)
{
    while (true)
    {
        for (int i = 0; i < DEMO_ITEMS; i++)
        {
            lcd->cls();
            lcd->backlight(ON);
            wait(1);

            (*menu_funcs[i])(lcd);

            wait(3);

            lcd->backlight(OFF);
            wait(3);
        }
    }
}
*/

// Display the simple graphics demo
void graphics(N3310LCD* lcd, Joystick* jstick)
{
    lcd->writeString( 0, 1, "Text Demo", NORMAL);
    lcd->writeString(24, 5, "START", HIGHLIGHT );
    checkKeypressed(jstick);
    lcd->cls();

    lcd->writeStringBig( 0, 0, "123456", NORMAL );
    lcd->writeStringBig( 0, 3, "7890+-.", NORMAL );
    wait(2);

    lcd->writeStringBig( 0, 0, "123456", HIGHLIGHT );
    lcd->writeStringBig( 0, 3, "7890+-.", HIGHLIGHT );
    wait(2);

    lcd->cls();

    lcd->writeString( 0, 1, "Graphic Demo", NORMAL);
    lcd->writeString(24, 5, "START", HIGHLIGHT );
    checkKeypressed(jstick);
    lcd->cls();
    // Draw some circles pulsing in and out
    for(int a=0; a< 4; a++) {
        for( int r = 1; r < 49; r+=1 ) {
            lcd->drawCircle(42, 24, r, PIXEL_ON );
            wait(0.010);
        }
        wait(0.010);
        for( int r = 48; r >0; r-=1 ) {
            wait(0.010);
            lcd->drawCircle(42, 24, r, PIXEL_OFF );
        }
    }

    // More circles
    for( int xc = 10; xc < 85; xc+=15 ) {
        lcd->drawCircle(xc, 24, 20, PIXEL_ON );
    }
    wait( 2 );

    // Draw diagonal lines using XOR colour
    lcd->drawLine(0,0,83,47, PIXEL_XOR);
    lcd->drawLine(0,43,83,0, PIXEL_XOR);

    wait( 2 );

    // Draw a rectangle
    lcd->drawRectangle(5,5,78,42, PIXEL_ON);

    wait( 2 );

    // Draw 2 filled rectangles
    lcd->drawFilledRectangle(5,3,78,20, PIXEL_ON);
    lcd->drawFilledRectangle(5,25,78,42, PIXEL_ON);

    wait( 2 );

    // Draw bitmap image
    lcd->drawBitmap( 0,0, fy1,84,48);
    wait(5);

    // Pacman animation
    lcd->cls();
    int px = 0;
    int py = 1;
    float pause=0.060;
    for( int p=0; p <9; p++) {
        lcd->drawBitmap( px,py, pacman1,32,32);
        wait( pause );
        lcd->clearBitmap( px++,py, 32,32);
        lcd->drawBitmap( px,py, pacman2,32,32);
        wait( pause );
        lcd->clearBitmap( px++,py, 32,32);
        lcd->drawBitmap( px,py, pacman3,32,32);
        wait( pause );
        lcd->clearBitmap( px++,py, 32,32);
        lcd->drawBitmap( px,py, pacman4,32,32);
        wait( pause );
        lcd->clearBitmap( px++,py, 32,32);
        lcd->drawBitmap( px,py, pacman3,32,32);
        wait( pause );
        lcd->clearBitmap( px++,py, 32,32);
        lcd->drawBitmap( px,py, pacman2,32,32);
        wait( pause );
        lcd->clearBitmap( px++,py, 32,32);
    }
    lcd->drawBitmap( px,py, pacman1,32,32);

    wait( 5 );

    lcd->cls();

    lcd->writeString( 0, 1, "Graphic Demo", NORMAL);
    lcd->writeString( 0, 3, "The End!!", NORMAL);
//  lcd->writeString(38, 5, "OK", HIGHLIGHT );
    waitforOKKey(lcd,jstick);
}


int main()
{
    Joystick jstick(N3310SPIPort::AD0);
    N3310LCD lcd(N3310SPIPort::MOSI, N3310SPIPort::MISO, N3310SPIPort::SCK,
                 N3310SPIPort::CE, N3310SPIPort::DAT_CMD, N3310SPIPort::LCD_RST,
                 N3310SPIPort::BL_ON);
    lcd.init();
    lcd.cls();
    lcd.backlight(ON);

    // demo stuff
    // autoDemo(&lcd);

    initMenu(&lcd);
    int currentMenuItem = 0;
    Ticker jstickPoll;
    jstickPoll.attach(&jstick, &Joystick::updateADCKey, 0.01);    // check ever 10ms


    while (true) {
        for (int i = 0; i < NUM_KEYS; i++) {
            if (jstick.getKeyState(i) != 0) {
                jstick.resetKeyState(i);  // reset button flag
                switch(i) {
                    case UP_KEY:
                        // current item to normal display
                        lcd.writeString(MENU_X, MENU_Y + currentMenuItem, menu_items[currentMenuItem], NORMAL);
                        currentMenuItem -=1;
                        if (currentMenuItem <0)  currentMenuItem = DEMO_ITEMS -1;
                        // next item to highlight display
                        lcd.writeString(MENU_X, MENU_Y + currentMenuItem, menu_items[currentMenuItem], HIGHLIGHT);
                        break;

                    case DOWN_KEY:
                        // current item to normal display
                        lcd.writeString(MENU_X, MENU_Y + currentMenuItem, menu_items[currentMenuItem], NORMAL);
                        currentMenuItem +=1;
                        if(currentMenuItem >(DEMO_ITEMS - 1))  currentMenuItem = 0;
                        // next item to highlight display
                        lcd.writeString(MENU_X, MENU_Y + currentMenuItem, menu_items[currentMenuItem], HIGHLIGHT);
                        break;

                    case LEFT_KEY:
                        initMenu(&lcd);
                        currentMenuItem = 0;
                        break;

                    case RIGHT_KEY:
                        lcd.cls();
                        (*menu_funcs[currentMenuItem])(&lcd, &jstick);
                        waitforOKKey(&lcd, &jstick);
                        lcd.cls();
                        initMenu(&lcd);
                        currentMenuItem = 0;
                        break;
                }
            }
        }
    }


}
