/**
 * @section DESCRIPTION
 * N3310LCD. A program to interface mbed with the nuelectronics
 * Nokia 3310 LCD shield from www.nuelectronics.com. Ported from
 * the nuelectronics Arduino code.
 *
 * @section LICENSE
 *
 * Copyright (C) <2009> Petras Saduikis <petras@petras.co.uk>
 *
 * Converted to a mbed library by Andrew D. Lindsay
 *
 * This file is part of N3310LCD.
 *
 * N3310LCD is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * N3310LCD is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with N3310LCD.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef N3310LCD_H
#define N3310LCD_H

#include <mbed.h>
#include "N3310LCDDefs.h"

#define LCDCOLMAX   84
#define LCDROWMAX   6
#define LCDPIXELROWMAX  48
#define PIXEL_OFF 0
#define PIXEL_ON  1
#define PIXEL_XOR 2
#define FONT_5x7  0
#define FONT_6x8  1
#define FONT_ALPHA_17x17 2


class N3310LCD : public Stream
{
public:
    /** Constructor
     *
     * @param mosi Pin name
     * @param miso pin name
     * @param sck Clock pin name
     * @param cd Chip enable pin name
     * @param dat_cmd Data/Command selection pin
     * @param lcd_rst Reset pin name
     * @param bl_on Backlight control pin
     */
    N3310LCD(PinName mosi, PinName miso, PinName sck,
             PinName ce, PinName dat_cmd, PinName lcd_rst, PinName bl_on);

    /** Main initialisation function
     */
    void init();

    /** Clear LCD screen and home cursor
     */
    void cls();

    /** Set backlight state, either on or off
     *
     * @param state The backlight state, ON or OFF
     */
    void backlight(eBacklight state);

    /** Write a single command to the LCD module
     *
     * @param data The command data to send
     */
    void writeCommand(BYTE data);
    
    /** Write a byte of data to LCD module
     *
     * @param data The data byte to send
     */
    void writeData(BYTE data);
    
    /** Move the cursor to a particular location
     *
     * @param xPos X position, 0 to 83, left to right
     * @param yPos Y position, 0 to 47, top to bottom
     */
    void locate(BYTE xPos, BYTE yPos);

    /** Copy bitmap data to display, can be smaller than full screen for partial updates
     *
     * @param xPos X position, 0 to 83
     * @param yPos Y position, 0 to 47
     * @param bitmap Bitmap data 
     * @param bmpXSise Number of X pixels in bitmap
     * @param bmpYSize Number of Y pixels in bitmap
     */
    void drawBitmap(BYTE xPos, BYTE yPos, BYTE* bitmap, BYTE bmpXSize, BYTE bmpYSize);
    
    /** Clear an area of the display 
     *
     * @param xPos X position, 0 to 83
     * @param yPos Y position, 0 to 47
     * @param bmpXSise Number of X pixels in bitmap
     * @param bmpYSize Number of Y pixels in bitmap
     */
    void clearBitmap(BYTE xPos,BYTE yPos, BYTE size_x, BYTE size_y);
    
    /** Set the font to use for future characters
     *
     * @param fornt The font to use
     */
    void setFont(BYTE font );
    
    /** Write a string of chars to the display starting at specified position
     * and using specified display mode
     * 
     * @param xPos X position, 0 to 83
     * @param yPos Y position, 0 to 47
     * @param string The string to display 
     * @param mode NORMAL or HIGHLIGHT used to diplay text
     */
    void writeString(BYTE xPos, BYTE yPos, char* string, eDisplayMode mode);
    
    /** Write a string of chars to the display starting at specified position
     * and using specified display mode in the big font
     *
     * @param xPos X position, 0 to 83
     * @param yPos Y position, 0 to 47
     * @param string The string to display 
     * @param mode NORMAL or HIGHLIGHT used to diplay text
     */
    void writeStringBig(BYTE xPos, BYTE yPos, char* string, eDisplayMode mode);
    
    /** Write a single character to display in specified mode
     *
     * @param ch The character to display 
     * @param mode NORMAL or HIGHLIGHT used to diplay text
     */     
    void writeChar(BYTE ch, eDisplayMode mode);
    
    /** Write a single big character to display in specified mode
     *
     * @param xPos X position, 0 to 83
     * @param yPos Y position, 0 to 47
     * @param ch The character to display 
     * @param mode NORMAL or HIGHLIGHT used to diplay text
     */     
    void writeCharBig(BYTE xPos, BYTE yPos, BYTE ch, eDisplayMode mode);

    /** Set a single pixel either on or off
     *
     * @param x The X position, 0 to 83
     * @param y The Y position, 0 to 47
     * @param c The state, on or off.
     */
    void setPixel( BYTE x, BYTE y, BYTE c );
    
    /** Draw a line from one point to another in the specified colour
     *
     * @param x1 Origin X
     * @param y1 Origin Y
     * @param x2 Destination X
     * @param y2 Destination Y
     * @param c Colour, either PIXEL_ON, PIXEL_OFF or PIXEL_XOR
     */
    void drawLine(BYTE x1, BYTE y1, BYTE x2, BYTE y2, BYTE c);
    
    /** Draw a rectangle by specifying opposite corners. Can be on or off to draw or clear.
     *
     * @param x1 Origin X
     * @param y1 Origin Y
     * @param x2 Destination X
     * @param y2 Destination Y
     * @param c Colour, either PIXEL_ON, PIXEL_OFF or PIXEL_XOR
     */
    void drawRectangle(BYTE x1, BYTE y1,BYTE x2, BYTE y2, BYTE c);

    /** Draw a solid filled rectangle by specifying opposite corners. Can be on or off to draw or clear.
     *
     * @param x1 Origin X
     * @param y1 Origin Y
     * @param x2 Destination X
     * @param y2 Destination Y
     * @param c Colour, either PIXEL_ON, PIXEL_OFF or PIXEL_XOR
     */
    void drawFilledRectangle(BYTE x1, BYTE y1, BYTE x2, BYTE y2, BYTE c);
    
    /** Draw a circle of a given radius. Points outside of the display are not plotted allowing 
     * circles that go outside of display to be drawn.
     *
     * @param x1 Centre X
     * @param y1 Centre Y
     * @param r Radius length
     * @param c Colour, either PIXEL_ON, PIXEL_OFF or PIXEL_XOR
     */
    void drawCircle(BYTE xc, BYTE yc, BYTE r, BYTE c);

protected:
    /** Implementation of virtual functions used by printf stream
     */
    virtual int _putc(int value);
    /** Dummy for stream read
     */
    virtual int _getc();

private:
    // I/O
    SPI lcdPort;            // does SPI MOSI, MISO and SCK
    DigitalOut ceWire;      // does SPI CE
    DigitalOut dcWire;      // does 3310 DAT_CMD
    DigitalOut rstWire;     // does 3310 LCD_RST
    DigitalOut blWire;      // does 3310 BL_ON (backlight)
};

#endif